<?php
/**
 * @see       https://github.com/zendframework/zend-hydrator for the canonical source repository
 * @copyright Copyright (c) 2010-2018 Zend Technologies USA Inc. (https://www.zend.com)
 * @license   https://github.com/zendframework/zend-hydrator/blob/master/LICENSE.md New BSD License
 */

declare(strict_types=1);

namespace ZendTest\Hydrator;

use PHPUnit\Framework\TestCase;
use Zend\Hydrator\ArraySerializableHydrator;
use Zend\Hydrator\ClassMethodsHydrator;
use Zend\Hydrator\Filter\FilterComposite;
use Zend\Hydrator\ObjectPropertyHydrator;
use Zend\Hydrator\ReflectionHydrator;
use Zend\Hydrator\Strategy\DefaultStrategy;
use Zend\Hydrator\Strategy\SerializableStrategy;
use ZendTest\Hydrator\TestAsset\ArraySerializable as ArraySerializableAsset;
use ZendTest\Hydrator\TestAsset\ClassMethodsCamelCase;
use ZendTest\Hydrator\TestAsset\ClassMethodsCamelCaseMissing;
use ZendTest\Hydrator\TestAsset\ClassMethodsFilterProviderInterface;
use ZendTest\Hydrator\TestAsset\ClassMethodsInvalidParameter;
use ZendTest\Hydrator\TestAsset\ClassMethodsMagicMethodSetter;
use ZendTest\Hydrator\TestAsset\ClassMethodsProtectedSetter;
use ZendTest\Hydrator\TestAsset\ClassMethodsTitleCase;
use ZendTest\Hydrator\TestAsset\ClassMethodsUnderscore;
use ZendTest\Hydrator\TestAsset\ObjectProperty as ObjectPropertyAsset;
use ZendTest\Hydrator\TestAsset\Reflection as ReflectionAsset;
use ZendTest\Hydrator\TestAsset\ReflectionFilter;

use function explode;
use function get_class;
use function strlen;

class HydratorTest extends TestCase
{
    /**
     * @var ClassMethodsCamelCase
     */
    protected $classMethodsCamelCase;

    /**
     * @var ClassMethodsTitleCase
     */
    protected $classMethodsTitleCase;

    /**
     * @var ClassMethodsCamelCaseMissing
     */
    protected $classMethodsCamelCaseMissing;

    /**
     * @var ClassMethodsUnderscore
     */
    protected $classMethodsUnderscore;

    /**
     * @var ClassMethodsInvalidParameter
     */
    protected $classMethodsInvalidParameter;

    /**
     * @var ReflectionAsset
     */
    protected $reflection;

    protected function setUp()
    {
        $this->classMethodsCamelCase = new ClassMethodsCamelCase();
        $this->classMethodsTitleCase = new ClassMethodsTitleCase();
        $this->classMethodsCamelCaseMissing = new ClassMethodsCamelCaseMissing();
        $this->classMethodsUnderscore = new ClassMethodsUnderscore();
        $this->classMethodsInvalidParameter = new ClassMethodsInvalidParameter();
        $this->reflection = new ReflectionAsset;
        $this->classMethodsInvalidParameter = new ClassMethodsInvalidParameter();
    }

    public function testInitiateValues()
    {
        $this->assertEquals($this->classMethodsCamelCase->getFooBar(), '1');
        $this->assertEquals($this->classMethodsCamelCase->getFooBarBaz(), '2');
        $this->assertEquals($this->classMethodsCamelCase->getIsFoo(), true);
        $this->assertEquals($this->classMethodsCamelCase->isBar(), true);
        $this->assertEquals($this->classMethodsCamelCase->getHasFoo(), true);
        $this->assertEquals($this->classMethodsCamelCase->hasBar(), true);
        $this->assertEquals($this->classMethodsTitleCase->getFooBar(), '1');
        $this->assertEquals($this->classMethodsTitleCase->getFooBarBaz(), '2');
        $this->assertEquals($this->classMethodsTitleCase->getIsFoo(), true);
        $this->assertEquals($this->classMethodsTitleCase->getIsBar(), true);
        $this->assertEquals($this->classMethodsTitleCase->getHasFoo(), true);
        $this->assertEquals($this->classMethodsTitleCase->getHasBar(), true);
        $this->assertEquals($this->classMethodsUnderscore->getFooBar(), '1');
        $this->assertEquals($this->classMethodsUnderscore->getFooBarBaz(), '2');
        $this->assertEquals($this->classMethodsUnderscore->getIsFoo(), true);
        $this->assertEquals($this->classMethodsUnderscore->isBar(), true);
        $this->assertEquals($this->classMethodsUnderscore->getHasFoo(), true);
        $this->assertEquals($this->classMethodsUnderscore->hasBar(), true);
    }

    public function testHydratorReflection()
    {
        $hydrator = new ReflectionHydrator();
        $datas    = $hydrator->extract($this->reflection);
        $this->assertTrue(isset($datas['foo']));
        $this->assertEquals($datas['foo'], '1');
        $this->assertTrue(isset($datas['fooBar']));
        $this->assertEquals($datas['fooBar'], '2');
        $this->assertTrue(isset($datas['fooBarBaz']));
        $this->assertEquals($datas['fooBarBaz'], '3');

        $test = $hydrator->hydrate(['foo' => 'foo', 'fooBar' => 'bar', 'fooBarBaz' => 'baz'], $this->reflection);
        $this->assertEquals($test->foo, 'foo');
        $this->assertEquals($test->getFooBar(), 'bar');
        $this->assertEquals($test->getFooBarBaz(), 'baz');
    }

    public function testHydratorClassMethodsCamelCase()
    {
        $hydrator = new ClassMethodsHydrator(false);
        $datas = $hydrator->extract($this->classMethodsCamelCase);
        $this->assertTrue(isset($datas['fooBar']));
        $this->assertEquals($datas['fooBar'], '1');
        $this->assertTrue(isset($datas['fooBarBaz']));
        $this->assertFalse(isset($datas['foo_bar']));
        $this->assertTrue(isset($datas['isFoo']));
        $this->assertEquals($datas['isFoo'], true);
        $this->assertTrue(isset($datas['isBar']));
        $this->assertEquals($datas['isBar'], true);
        $this->assertTrue(isset($datas['hasFoo']));
        $this->assertEquals($datas['hasFoo'], true);
        $this->assertTrue(isset($datas['hasBar']));
        $this->assertEquals($datas['hasBar'], true);
        $test = $hydrator->hydrate(
            [
                'fooBar' => 'foo',
                'fooBarBaz' => 'bar',
                'isFoo' => false,
                'isBar' => false,
                'hasFoo' => false,
                'hasBar' => false,
            ],
            $this->classMethodsCamelCase
        );
        $this->assertSame($this->classMethodsCamelCase, $test);
        $this->assertEquals($test->getFooBar(), 'foo');
        $this->assertEquals($test->getFooBarBaz(), 'bar');
        $this->assertEquals($test->getIsFoo(), false);
        $this->assertEquals($test->isBar(), false);
        $this->assertEquals($test->getHasFoo(), false);
        $this->assertEquals($test->hasBar(), false);
    }

    public function testHydratorClassMethodsTitleCase()
    {
        $hydrator = new ClassMethodsHydrator(false);
        $datas = $hydrator->extract($this->classMethodsTitleCase);
        $this->assertTrue(isset($datas['FooBar']));
        $this->assertEquals($datas['FooBar'], '1');
        $this->assertTrue(isset($datas['FooBarBaz']));
        $this->assertFalse(isset($datas['foo_bar']));
        $this->assertTrue(isset($datas['IsFoo']));
        $this->assertEquals($datas['IsFoo'], true);
        $this->assertTrue(isset($datas['IsBar']));
        $this->assertEquals($datas['IsBar'], true);
        $this->assertTrue(isset($datas['HasFoo']));
        $this->assertEquals($datas['HasFoo'], true);
        $this->assertTrue(isset($datas['HasBar']));
        $this->assertEquals($datas['HasBar'], true);
        $test = $hydrator->hydrate(
            [
                    'FooBar' => 'foo',
                    'FooBarBaz' => 'bar',
                    'IsFoo' => false,
                    'IsBar' => false,
                    'HasFoo' => false,
                    'HasBar' => false,
            ],
            $this->classMethodsTitleCase
        );
        $this->assertSame($this->classMethodsTitleCase, $test);
        $this->assertEquals($test->getFooBar(), 'foo');
        $this->assertEquals($test->getFooBarBaz(), 'bar');
        $this->assertEquals($test->getIsFoo(), false);
        $this->assertEquals($test->getIsBar(), false);
        $this->assertEquals($test->getHasFoo(), false);
        $this->assertEquals($test->getHasBar(), false);
    }

    public function testHydratorClassMethodsUnderscore()
    {
        $hydrator = new ClassMethodsHydrator(true);
        $datas = $hydrator->extract($this->classMethodsUnderscore);
        $this->assertTrue(isset($datas['foo_bar']));
        $this->assertEquals($datas['foo_bar'], '1');
        $this->assertTrue(isset($datas['foo_bar_baz']));
        $this->assertFalse(isset($datas['fooBar']));
        $this->assertTrue(isset($datas['is_foo']));
        $this->assertFalse(isset($datas['isFoo']));
        $this->assertEquals($datas['is_foo'], true);
        $this->assertTrue(isset($datas['is_bar']));
        $this->assertFalse(isset($datas['isBar']));
        $this->assertEquals($datas['is_bar'], true);
        $this->assertTrue(isset($datas['has_foo']));
        $this->assertFalse(isset($datas['hasFoo']));
        $this->assertEquals($datas['has_foo'], true);
        $this->assertTrue(isset($datas['has_bar']));
        $this->assertFalse(isset($datas['hasBar']));
        $this->assertEquals($datas['has_bar'], true);
        $test = $hydrator->hydrate(
            [
                'foo_bar' => 'foo',
                'foo_bar_baz' => 'bar',
                'is_foo' => false,
                'is_bar' => false,
                'has_foo' => false,
                'has_bar' => false,
            ],
            $this->classMethodsUnderscore
        );
        $this->assertSame($this->classMethodsUnderscore, $test);
        $this->assertEquals($test->getFooBar(), 'foo');
        $this->assertEquals($test->getFooBarBaz(), 'bar');
        $this->assertEquals($test->getIsFoo(), false);
        $this->assertEquals($test->isBar(), false);
        $this->assertEquals($test->getHasFoo(), false);
        $this->assertEquals($test->hasBar(), false);
    }

    public function testHydratorClassMethodsUnderscoreWithUnderscoreUpperCasedHydrateDataKeys()
    {
        $hydrator = new ClassMethodsHydrator(true);
        $datas = $hydrator->extract($this->classMethodsUnderscore);
        $test = $hydrator->hydrate(
            [
                'FOO_BAR' => 'foo',
                'FOO_BAR_BAZ' => 'bar',
                'IS_FOO' => false,
                'IS_BAR' => false,
                'HAS_FOO' => false,
                'HAS_BAR' => false,
            ],
            $this->classMethodsUnderscore
        );
        $this->assertSame($this->classMethodsUnderscore, $test);
        $this->assertEquals($test->getFooBar(), 'foo');
        $this->assertEquals($test->getFooBarBaz(), 'bar');
        $this->assertEquals($test->getIsFoo(), false);
        $this->assertEquals($test->isBar(), false);
        $this->assertEquals($test->getHasFoo(), false);
        $this->assertEquals($test->hasBar(), false);
    }

    public function testHydratorClassMethodsOptions()
    {
        $hydrator = new ClassMethodsHydrator();
        $this->assertTrue($hydrator->getUnderscoreSeparatedKeys());
        $hydrator->setOptions(['underscoreSeparatedKeys' => false]);
        $this->assertFalse($hydrator->getUnderscoreSeparatedKeys());
        $hydrator->setUnderscoreSeparatedKeys(true);
        $this->assertTrue($hydrator->getUnderscoreSeparatedKeys());
    }

    public function testHydratorClassMethodsIgnoresInvalidValues()
    {
        $hydrator = new ClassMethodsHydrator(true);
        $data = [
            'foo_bar' => '1',
            'foo_bar_baz' => '2',
            'invalid' => 'value'
        ];
        $test = $hydrator->hydrate($data, $this->classMethodsUnderscore);
        $this->assertSame($this->classMethodsUnderscore, $test);
    }

    public function testHydratorClassMethodsDefaultBehaviorIsConvertUnderscoreToCamelCase()
    {
        $hydrator = new ClassMethodsHydrator();
        $datas = $hydrator->extract($this->classMethodsUnderscore);
        $this->assertTrue(isset($datas['foo_bar']));
        $this->assertEquals($datas['foo_bar'], '1');
        $this->assertTrue(isset($datas['foo_bar_baz']));
        $this->assertFalse(isset($datas['fooBar']));
        $test = $hydrator->hydrate(['foo_bar' => 'foo', 'foo_bar_baz' => 'bar'], $this->classMethodsUnderscore);
        $this->assertSame($this->classMethodsUnderscore, $test);
        $this->assertEquals($test->getFooBar(), 'foo');
        $this->assertEquals($test->getFooBarBaz(), 'bar');
    }

    public function testRetrieveWildStrategyAndOther()
    {
        $hydrator = new ClassMethodsHydrator();
        $hydrator->addStrategy('default', new DefaultStrategy());
        $hydrator->addStrategy('*', new SerializableStrategy('phpserialize'));
        $default = $hydrator->getStrategy('default');
        $this->assertEquals(get_class($default), DefaultStrategy::class);
        $serializable = $hydrator->getStrategy('*');
        $this->assertEquals(get_class($serializable), SerializableStrategy::class);
    }

    public function testUseWildStrategyByDefault()
    {
        $hydrator = new ClassMethodsHydrator();
        $datas = $hydrator->extract($this->classMethodsUnderscore);
        $this->assertEquals($datas['foo_bar'], '1');
        $hydrator->addStrategy('*', new SerializableStrategy('phpserialize'));
        $datas = $hydrator->extract($this->classMethodsUnderscore);
        $this->assertEquals($datas['foo_bar'], 's:1:"1";');
    }

    public function testUseWildStrategyAndOther()
    {
        $hydrator = new ClassMethodsHydrator();
        $datas = $hydrator->extract($this->classMethodsUnderscore);
        $this->assertEquals($datas['foo_bar'], '1');
        $hydrator->addStrategy('foo_bar', new DefaultStrategy());
        $hydrator->addStrategy('*', new SerializableStrategy('phpserialize'));
        $datas = $hydrator->extract($this->classMethodsUnderscore);
        $this->assertEquals($datas['foo_bar'], '1');
        $this->assertEquals($datas['foo_bar_baz'], 's:1:"2";');
    }

    public function testHydratorClassMethodsCamelCaseWithSetterMissing()
    {
        $hydrator = new ClassMethodsHydrator(false);

        $datas = $hydrator->extract($this->classMethodsCamelCaseMissing);
        $this->assertTrue(isset($datas['fooBar']));
        $this->assertEquals($datas['fooBar'], '1');
        $this->assertTrue(isset($datas['fooBarBaz']));
        $this->assertFalse(isset($datas['foo_bar']));
        $test = $hydrator->hydrate(['fooBar' => 'foo', 'fooBarBaz' => 1], $this->classMethodsCamelCaseMissing);
        $this->assertSame($this->classMethodsCamelCaseMissing, $test);
        $this->assertEquals($test->getFooBar(), 'foo');
        $this->assertEquals($test->getFooBarBaz(), '2');
    }

    public function testHydratorClassMethodsManipulateFilter()
    {
        $hydrator = new ClassMethodsHydrator(false);
        $datas = $hydrator->extract($this->classMethodsCamelCase);

        $this->assertTrue(isset($datas['fooBar']));
        $this->assertEquals($datas['fooBar'], '1');
        $this->assertTrue(isset($datas['fooBarBaz']));
        $this->assertFalse(isset($datas['foo_bar']));
        $this->assertTrue(isset($datas['isFoo']));
        $this->assertEquals($datas['isFoo'], true);
        $this->assertTrue(isset($datas['isBar']));
        $this->assertEquals($datas['isBar'], true);
        $this->assertTrue(isset($datas['hasFoo']));
        $this->assertEquals($datas['hasFoo'], true);
        $this->assertTrue(isset($datas['hasBar']));
        $this->assertEquals($datas['hasBar'], true);

        $hydrator->removeFilter('has');
        $datas = $hydrator->extract($this->classMethodsCamelCase);
        $this->assertTrue(isset($datas['hasFoo'])); //method is getHasFoo
        $this->assertFalse(isset($datas['hasBar'])); //method is hasBar
    }

    public function testHydratorClassMethodsWithCustomFilter()
    {
        $hydrator = new ClassMethodsHydrator(false);
        $datas = $hydrator->extract($this->classMethodsCamelCase);
        $hydrator->addFilter(
            "exclude",
            function ($property) {
                list($class, $method) = explode('::', $property);

                if ($method == 'getHasFoo') {
                    return false;
                }

                return true;
            },
            FilterComposite::CONDITION_AND
        );

        $datas = $hydrator->extract($this->classMethodsCamelCase);
        $this->assertFalse(isset($datas['hasFoo']));
    }

    /**
     * @dataProvider filterProvider
     */
    public function testArraySerializableFilter($hydrator, $serializable)
    {
        $this->assertSame(
            [
                "foo" => "bar",
                "bar" => "foo",
                "blubb" => "baz",
                "quo" => "blubb"
            ],
            $hydrator->extract($serializable)
        );

        $hydrator->addFilter("foo", function ($property) {
            if ($property == "foo") {
                return false;
            }
            return true;
        });

        $this->assertSame(
            [
                "bar" => "foo",
                "blubb" => "baz",
                "quo" => "blubb"
            ],
            $hydrator->extract($serializable)
        );

        $hydrator->addFilter("len", function ($property) {
            if (strlen($property) !== 3) {
                return false;
            }
            return true;
        }, FilterComposite::CONDITION_AND);

        $this->assertSame(
            [
                "bar" => "foo",
                "quo" => "blubb"
            ],
            $hydrator->extract($serializable)
        );

        $hydrator->removeFilter("len");
        $hydrator->removeFilter("foo");

        $this->assertSame(
            [
                "foo" => "bar",
                "bar" => "foo",
                "blubb" => "baz",
                "quo" => "blubb"
            ],
            $hydrator->extract($serializable)
        );
    }

    public function filterProvider()
    {
        return [
            [new ObjectPropertyHydrator(), new ObjectPropertyAsset],
            [new ArraySerializableHydrator(), new ArraySerializableAsset],
            [new ReflectionHydrator(), new ReflectionFilter]
        ];
    }

    public function testHydratorClassMethodsWithInvalidNumberOfParameters()
    {
        $hydrator = new ClassMethodsHydrator(false);
        $datas = $hydrator->extract($this->classMethodsInvalidParameter);

        $this->assertTrue($datas['hasBar']);
        $this->assertEquals('Bar', $datas['foo']);
        $this->assertFalse($datas['isBla']);
    }

    public function testObjectBasedFilters()
    {
        $hydrator = new ClassMethodsHydrator(false);
        $foo = new ClassMethodsFilterProviderInterface();
        $data = $hydrator->extract($foo);
        $this->assertArrayNotHasKey("filter", $data);
        $this->assertSame("bar", $data["foo"]);
        $this->assertSame("foo", $data["bar"]);
    }

    public function testHydratorClassMethodsWithProtectedSetter()
    {
        $hydrator = new ClassMethodsHydrator(false);
        $object = new ClassMethodsProtectedSetter();
        $hydrator->hydrate(['foo' => 'bar', 'bar' => 'BAR'], $object);
        $data = $hydrator->extract($object);

        $this->assertEquals($data['bar'], 'BAR');
    }

    public function testHydratorClassMethodsWithMagicMethodSetter()
    {
        $hydrator = new ClassMethodsHydrator(false);
        $object = new ClassMethodsMagicMethodSetter();
        $hydrator->hydrate(['foo' => 'bar'], $object);
        $data = $hydrator->extract($object);

        $this->assertEquals($data['foo'], 'bar');
    }

    public function testHydratorClassMethodsWithMagicMethodSetterAndMethodExistsCheck()
    {
        $hydrator = new ClassMethodsHydrator(false, true);
        $object = new ClassMethodsMagicMethodSetter();
        $hydrator->hydrate(['foo' => 'bar'], $object);
        $data = $hydrator->extract($object);

        $this->assertNull($data['foo']);
    }
}
