<?php
/**
 * PHPCompatibility, an external standard for PHP_CodeSniffer.
 *
 * @package   PHPCompatibility
 * @copyright 2012-2020 PHPCompatibility Contributors
 * @license   https://opensource.org/licenses/LGPL-3.0 LGPL3
 * @link      https://github.com/PHPCompatibility/PHPCompatibility
 */

namespace PHPCompatibility\Tests\Constants;

use PHPCompatibility\Tests\BaseSniffTest;

/**
 * Test the NewConstants sniff.
 *
 * @group newConstants
 * @group constants
 *
 * @covers \PHPCompatibility\Sniffs\Constants\NewConstantsSniff
 *
 * @since 8.1.0
 */
class NewConstantsUnitTest extends BaseSniffTest
{

    /**
     * testNewConstant
     *
     * @dataProvider dataNewConstant
     *
     * @param string $constantName      Name of the PHP constant.
     * @param string $lastVersionBefore The PHP version just *before* the constant was introduced.
     * @param array  $lines             The line numbers in the test file which apply to this constant.
     * @param string $okVersion         A PHP version in which the constant was valid.
     * @param string $testVersion       Optional. A PHP version in which to test for the error if different
     *                                  from the $lastVersionBefore.
     *
     * @return void
     */
    public function testNewConstant($constantName, $lastVersionBefore, $lines, $okVersion, $testVersion = null)
    {
        $errorVersion = (isset($testVersion)) ? $testVersion : $lastVersionBefore;
        $file         = $this->sniffFile(__FILE__, $errorVersion);
        $error        = "The constant \"{$constantName}\" is not present in PHP version {$lastVersionBefore} or earlier";
        foreach ($lines as $line) {
            $this->assertError($file, $line, $error);
        }

        $file = $this->sniffFile(__FILE__, $okVersion);
        foreach ($lines as $line) {
            $this->assertNoViolation($file, $line);
        }
    }

    /**
     * Data provider.
     *
     * @see testNewConstant()
     *
     * @return array
     */
    public function dataNewConstant()
    {
        return [
            ['E_STRICT', '4.4', [8], '5.0'],
            ['CURLOPT_FTP_USE_EPRT', '4.4', [9], '5.0'],
            ['CURLOPT_NOSIGNAL', '4.4', [10], '5.0'],
            ['CURLOPT_UNRESTRICTED_AUTH', '4.4', [11], '5.0'],
            ['CURLOPT_BUFFERSIZE', '4.4', [12], '5.0'],
            ['CURLOPT_HTTPAUTH', '4.4', [13], '5.0'],
            ['CURLOPT_PROXYPORT', '4.4', [14], '5.0'],
            ['CURLOPT_PROXYTYPE', '4.4', [15], '5.0'],
            ['CURLOPT_SSLCERTTYPE', '4.4', [16], '5.0'],
            ['CURLOPT_HTTP200ALIASES', '4.4', [17], '5.0'],
            ['M_PI', '4.4', [900], '5.0'],
            ['M_E', '4.4', [901], '5.0'],
            ['M_LOG2E', '4.4', [902], '5.0'],
            ['M_LOG10E', '4.4', [903], '5.0'],
            ['M_LN2', '4.4', [904], '5.0'],
            ['M_LN10', '4.4', [905], '5.0'],
            ['M_PI_2', '4.4', [906], '5.0'],
            ['M_PI_4', '4.4', [907], '5.0'],
            ['M_1_PI', '4.4', [908], '5.0'],
            ['M_2_PI', '4.4', [909], '5.0'],
            ['M_2_SQRTPI', '4.4', [910], '5.0'],
            ['M_SQRT2', '4.4', [911], '5.0'],
            ['M_SQRT1_2', '4.4', [912], '5.0'],
            ['NAN', '4.4', [913], '5.0'],
            ['INF', '4.4', [914], '5.0'],
            ['OPENSSL_ALGO_MD2', '4.4', [18], '5.0'],
            ['OPENSSL_ALGO_MD4', '4.4', [19], '5.0'],
            ['OPENSSL_ALGO_MD5', '4.4', [20], '5.0'],
            ['OPENSSL_ALGO_SHA1', '4.4', [21], '5.0'],
            ['OPENSSL_ALGO_DSS1', '4.4', [22], '5.0'],
            ['T_ABSTRACT', '4.4', [23], '5.0'],
            ['T_CATCH', '4.4', [24], '5.0'],
            ['TIDY_TAG_UNKNOWN', '4.4', [915], '5.0'],
            ['TIDY_TAG_A', '4.4', [916], '5.0'],
            ['TIDY_TAG_ABBR', '4.4', [917], '5.0'],
            ['TIDY_TAG_ACRONYM', '4.4', [918], '5.0'],
            ['TIDY_TAG_ALIGN', '4.4', [919], '5.0'],
            ['TIDY_TAG_APPLET', '4.4', [920], '5.0'],
            ['TIDY_TAG_AREA', '4.4', [921], '5.0'],
            ['TIDY_TAG_B', '4.4', [922], '5.0'],
            ['TIDY_TAG_BASE', '4.4', [923], '5.0'],
            ['TIDY_TAG_BASEFONT', '4.4', [924], '5.0'],
            ['TIDY_TAG_BDO', '4.4', [925], '5.0'],
            ['TIDY_TAG_BGSOUND', '4.4', [926], '5.0'],
            ['TIDY_TAG_BIG', '4.4', [927], '5.0'],
            ['TIDY_TAG_BLINK', '4.4', [928], '5.0'],
            ['TIDY_TAG_BLOCKQUOTE', '4.4', [929], '5.0'],
            ['TIDY_TAG_BODY', '4.4', [930], '5.0'],
            ['TIDY_TAG_BR', '4.4', [931], '5.0'],
            ['TIDY_TAG_BUTTON', '4.4', [932], '5.0'],
            ['TIDY_TAG_CAPTION', '4.4', [933], '5.0'],
            ['TIDY_TAG_CENTER', '4.4', [934], '5.0'],
            ['TIDY_TAG_CITE', '4.4', [935], '5.0'],
            ['TIDY_TAG_CODE', '4.4', [936], '5.0'],
            ['TIDY_TAG_COL', '4.4', [937], '5.0'],
            ['TIDY_TAG_COLGROUP', '4.4', [938], '5.0'],
            ['TIDY_TAG_COMMENT', '4.4', [939], '5.0'],
            ['TIDY_TAG_DD', '4.4', [940], '5.0'],
            ['TIDY_TAG_DEL', '4.4', [941], '5.0'],
            ['TIDY_TAG_DFN', '4.4', [942], '5.0'],
            ['TIDY_TAG_DIR', '4.4', [943], '5.0'],
            ['TIDY_TAG_DIV', '4.4', [944], '5.0'],
            ['TIDY_TAG_DL', '4.4', [945], '5.0'],
            ['TIDY_TAG_DT', '4.4', [946], '5.0'],
            ['TIDY_TAG_EM', '4.4', [947], '5.0'],
            ['TIDY_TAG_EMBED', '4.4', [948], '5.0'],
            ['TIDY_TAG_FIELDSET', '4.4', [949], '5.0'],
            ['TIDY_TAG_FONT', '4.4', [950], '5.0'],
            ['TIDY_TAG_FORM', '4.4', [951], '5.0'],
            ['TIDY_TAG_FRAME', '4.4', [952], '5.0'],
            ['TIDY_TAG_FRAMESET', '4.4', [953], '5.0'],
            ['TIDY_TAG_H1', '4.4', [954], '5.0'],
            ['TIDY_TAG_H2', '4.4', [955], '5.0'],
            ['TIDY_TAG_H3', '4.4', [956], '5.0'],
            ['TIDY_TAG_H4', '4.4', [957], '5.0'],
            ['TIDY_TAG_H5', '4.4', [958], '5.0'],
            ['TIDY_TAG_H6', '4.4', [959], '5.0'],
            ['TIDY_TAG_HEAD', '4.4', [960], '5.0'],
            ['TIDY_TAG_HR', '4.4', [961], '5.0'],
            ['TIDY_TAG_HTML', '4.4', [962], '5.0'],
            ['TIDY_TAG_I', '4.4', [963], '5.0'],
            ['TIDY_TAG_IFRAME', '4.4', [964], '5.0'],
            ['TIDY_TAG_ILAYER', '4.4', [965], '5.0'],
            ['TIDY_TAG_IMG', '4.4', [966], '5.0'],
            ['TIDY_TAG_INPUT', '4.4', [967], '5.0'],
            ['TIDY_TAG_INS', '4.4', [968], '5.0'],
            ['TIDY_TAG_ISINDEX', '4.4', [969], '5.0'],
            ['TIDY_TAG_KBD', '4.4', [970], '5.0'],
            ['TIDY_TAG_KEYGEN', '4.4', [971], '5.0'],
            ['TIDY_TAG_LABEL', '4.4', [972], '5.0'],
            ['TIDY_TAG_LAYER', '4.4', [973], '5.0'],
            ['TIDY_TAG_LEGEND', '4.4', [974], '5.0'],
            ['TIDY_TAG_LI', '4.4', [975], '5.0'],
            ['TIDY_TAG_LINK', '4.4', [976], '5.0'],
            ['TIDY_TAG_LISTING', '4.4', [977], '5.0'],
            ['TIDY_TAG_MAP', '4.4', [978], '5.0'],
            ['TIDY_TAG_MARQUEE', '4.4', [979], '5.0'],
            ['TIDY_TAG_MENU', '4.4', [980], '5.0'],
            ['TIDY_TAG_META', '4.4', [981], '5.0'],
            ['TIDY_TAG_MULTICOL', '4.4', [982], '5.0'],
            ['TIDY_TAG_NOBR', '4.4', [983], '5.0'],
            ['TIDY_TAG_NOEMBED', '4.4', [984], '5.0'],
            ['TIDY_TAG_NOFRAMES', '4.4', [985], '5.0'],
            ['TIDY_TAG_NOLAYER', '4.4', [986], '5.0'],
            ['TIDY_TAG_NOSAVE', '4.4', [987], '5.0'],
            ['TIDY_TAG_NOSCRIPT', '4.4', [988], '5.0'],
            ['TIDY_TAG_OBJECT', '4.4', [989], '5.0'],
            ['TIDY_TAG_OL', '4.4', [990], '5.0'],
            ['TIDY_TAG_OPTGROUP', '4.4', [991], '5.0'],
            ['TIDY_TAG_OPTION', '4.4', [992], '5.0'],
            ['TIDY_TAG_P', '4.4', [993], '5.0'],
            ['TIDY_TAG_PARAM', '4.4', [994], '5.0'],
            ['TIDY_TAG_PLAINTEXT', '4.4', [995], '5.0'],
            ['TIDY_TAG_PRE', '4.4', [996], '5.0'],
            ['TIDY_TAG_Q', '4.4', [997], '5.0'],
            ['TIDY_TAG_RB', '4.4', [998], '5.0'],
            ['TIDY_TAG_RBC', '4.4', [999], '5.0'],
            ['TIDY_TAG_RP', '4.4', [1000], '5.0'],
            ['TIDY_TAG_RT', '4.4', [1001], '5.0'],
            ['TIDY_TAG_RTC', '4.4', [1002], '5.0'],
            ['TIDY_TAG_RUBY', '4.4', [1003], '5.0'],
            ['TIDY_TAG_S', '4.4', [1004], '5.0'],
            ['TIDY_TAG_SAMP', '4.4', [1005], '5.0'],
            ['TIDY_TAG_SCRIPT', '4.4', [1006], '5.0'],
            ['TIDY_TAG_SELECT', '4.4', [1007], '5.0'],
            ['TIDY_TAG_SERVER', '4.4', [1008], '5.0'],
            ['TIDY_TAG_SERVLET', '4.4', [1009], '5.0'],
            ['TIDY_TAG_SMALL', '4.4', [1010], '5.0'],
            ['TIDY_TAG_SPACER', '4.4', [1011], '5.0'],
            ['TIDY_TAG_SPAN', '4.4', [1012], '5.0'],
            ['TIDY_TAG_STRIKE', '4.4', [1013], '5.0'],
            ['TIDY_TAG_STRONG', '4.4', [1014], '5.0'],
            ['TIDY_TAG_STYLE', '4.4', [1015], '5.0'],
            ['TIDY_TAG_SUB', '4.4', [1016], '5.0'],
            ['TIDY_TAG_SUP', '4.4', [1017], '5.0'],
            ['TIDY_TAG_TABLE', '4.4', [1018], '5.0'],
            ['TIDY_TAG_TBODY', '4.4', [1019], '5.0'],
            ['TIDY_TAG_TD', '4.4', [1020], '5.0'],
            ['TIDY_TAG_TEXTAREA', '4.4', [1021], '5.0'],
            ['TIDY_TAG_TFOOT', '4.4', [1022], '5.0'],
            ['TIDY_TAG_TH', '4.4', [1023], '5.0'],
            ['TIDY_TAG_THEAD', '4.4', [1024], '5.0'],
            ['TIDY_TAG_TITLE', '4.4', [1025], '5.0'],
            ['TIDY_TAG_TR', '4.4', [1026], '5.0'],
            ['TIDY_TAG_TT', '4.4', [1027], '5.0'],
            ['TIDY_TAG_U', '4.4', [1028], '5.0'],
            ['TIDY_TAG_UL', '4.4', [1029], '5.0'],
            ['TIDY_TAG_VAR', '4.4', [1030], '5.0'],
            ['TIDY_TAG_WBR', '4.4', [1031], '5.0'],
            ['TIDY_TAG_XMP', '4.4', [1032], '5.0'],
            ['TIDY_NODETYPE_ROOT', '4.4', [1033], '5.0'],
            ['TIDY_NODETYPE_DOCTYPE', '4.4', [1034], '5.0'],
            ['TIDY_NODETYPE_COMMENT', '4.4', [1035], '5.0'],
            ['TIDY_NODETYPE_PROCINS', '4.4', [1036], '5.0'],
            ['TIDY_NODETYPE_TEXT', '4.4', [1037], '5.0'],
            ['TIDY_NODETYPE_START', '4.4', [1038], '5.0'],
            ['TIDY_NODETYPE_END', '4.4', [1039], '5.0'],
            ['TIDY_NODETYPE_STARTEND', '4.4', [1040], '5.0'],
            ['TIDY_NODETYPE_CDATA', '4.4', [1041], '5.0'],
            ['TIDY_NODETYPE_SECTION', '4.4', [1042], '5.0'],
            ['TIDY_NODETYPE_ASP', '4.4', [1043], '5.0'],
            ['TIDY_NODETYPE_JSTE', '4.4', [1044], '5.0'],
            ['TIDY_NODETYPE_PHP', '4.4', [1045], '5.0'],
            ['TIDY_NODETYPE_XMLDECL', '4.4', [1046], '5.0'],
            ['SOAP_1_1', '4.4', [1114], '5.0'],
            ['SOAP_1_2', '4.4', [1115], '5.0'],
            ['SOAP_PERSISTENCE_SESSION', '4.4', [1116], '5.0'],
            ['SOAP_PERSISTENCE_REQUEST', '4.4', [1117], '5.0'],
            ['SOAP_FUNCTIONS_ALL', '4.4', [1118], '5.0'],
            ['SOAP_ENCODED', '4.4', [1119], '5.0'],
            ['SOAP_LITERAL', '4.4', [1120], '5.0'],
            ['SOAP_RPC', '4.4', [1121], '5.0'],
            ['SOAP_DOCUMENT', '4.4', [1122], '5.0'],
            ['SOAP_ACTOR_NEXT', '4.4', [1123], '5.0'],
            ['SOAP_ACTOR_NONE', '4.4', [1124], '5.0'],
            ['SOAP_ACTOR_UNLIMATERECEIVER', '4.4', [1125], '5.0'],
            ['SOAP_COMPRESSION_ACCEPT', '4.4', [1126], '5.0'],
            ['SOAP_COMPRESSION_GZIP', '4.4', [1127], '5.0'],
            ['SOAP_COMPRESSION_DEFLATE', '4.4', [1128], '5.0'],
            ['SOAP_AUTHENTICATION_BASIC', '4.4', [1129], '5.0'],
            ['SOAP_AUTHENTICATION_DIGEST', '4.4', [1130], '5.0'],
            ['UNKNOWN_TYPE', '4.4', [1131], '5.0'],
            ['XSD_STRING', '4.4', [1132], '5.0'],
            ['XSD_BOOLEAN', '4.4', [1133], '5.0'],
            ['XSD_DECIMAL', '4.4', [1134], '5.0'],
            ['XSD_FLOAT', '4.4', [1135], '5.0'],
            ['XSD_DOUBLE', '4.4', [1136], '5.0'],
            ['XSD_DURATION', '4.4', [1137], '5.0'],
            ['XSD_DATETIME', '4.4', [1138], '5.0'],
            ['XSD_TIME', '4.4', [1139], '5.0'],
            ['XSD_DATE', '4.4', [1140], '5.0'],
            ['XSD_GYEARMONTH', '4.4', [1141], '5.0'],
            ['XSD_GYEAR', '4.4', [1142], '5.0'],
            ['XSD_GMONTHDAY', '4.4', [1143], '5.0'],
            ['XSD_GDAY', '4.4', [1144], '5.0'],
            ['XSD_GMONTH', '4.4', [1145], '5.0'],
            ['XSD_HEXBINARY', '4.4', [1146], '5.0'],
            ['XSD_BASE64BINARY', '4.4', [1147], '5.0'],
            ['XSD_ANYURI', '4.4', [1148], '5.0'],
            ['XSD_QNAME', '4.4', [1149], '5.0'],
            ['XSD_NOTATION', '4.4', [1150], '5.0'],
            ['XSD_NORMALIZEDSTRING', '4.4', [1151], '5.0'],
            ['XSD_TOKEN', '4.4', [1152], '5.0'],
            ['XSD_LANGUAGE', '4.4', [1153], '5.0'],
            ['XSD_NMTOKEN', '4.4', [1154], '5.0'],
            ['XSD_NAME', '4.4', [1155], '5.0'],
            ['XSD_NCNAME', '4.4', [1156], '5.0'],
            ['XSD_ID', '4.4', [1157], '5.0'],
            ['XSD_IDREF', '4.4', [1158], '5.0'],
            ['XSD_IDREFS', '4.4', [1159], '5.0'],
            ['XSD_ENTITY', '4.4', [1160], '5.0'],
            ['XSD_ENTITIES', '4.4', [1161], '5.0'],
            ['XSD_INTEGER', '4.4', [1162], '5.0'],
            ['XSD_NONPOSITIVEINTEGER', '4.4', [1163], '5.0'],
            ['XSD_NEGATIVEINTEGER', '4.4', [1164], '5.0'],
            ['XSD_LONG', '4.4', [1165], '5.0'],
            ['XSD_INT', '4.4', [1166], '5.0'],
            ['XSD_SHORT', '4.4', [1167], '5.0'],
            ['XSD_BYTE', '4.4', [1168], '5.0'],
            ['XSD_NONNEGATIVEINTEGER', '4.4', [1169], '5.0'],
            ['XSD_UNSIGNEDLONG', '4.4', [1170], '5.0'],
            ['XSD_UNSIGNEDINT', '4.4', [1171], '5.0'],
            ['XSD_UNSIGNEDSHORT', '4.4', [1172], '5.0'],
            ['XSD_UNSIGNEDBYTE', '4.4', [1173], '5.0'],
            ['XSD_POSITIVEINTEGER', '4.4', [1174], '5.0'],
            ['XSD_NMTOKENS', '4.4', [1175], '5.0'],
            ['XSD_ANYTYPE', '4.4', [1176], '5.0'],
            ['XSD_ANYXML', '4.4', [1177], '5.0'],
            ['APACHE_MAP', '4.4', [1178], '5.0'],
            ['SOAP_ENC_OBJECT', '4.4', [1179], '5.0'],
            ['SOAP_ENC_ARRAY', '4.4', [1180], '5.0'],
            ['XSD_1999_TIMEINSTANT', '4.4', [1181], '5.0'],
            ['XSD_NAMESPACE', '4.4', [1182], '5.0'],
            ['XSD_1999_NAMESPACE', '4.4', [1183], '5.0'],
            ['SOAP_SINGLE_ELEMENT_ARRAYS', '4.4', [1184], '5.0'],
            ['SOAP_WAIT_ONE_WAY_CALLS', '4.4', [1185], '5.0'],
            ['SOAP_USE_XSI_ARRAY_TYPE', '4.4', [1186], '5.0'],
            ['WSDL_CACHE_NONE', '4.4', [1187], '5.0'],
            ['WSDL_CACHE_DISK', '4.4', [1188], '5.0'],
            ['WSDL_CACHE_MEMORY', '4.4', [1189], '5.0'],
            ['WSDL_CACHE_BOTH', '4.4', [1190], '5.0'],
            ['XML_ELEMENT_NODE', '4.4', [1197], '5.0'],
            ['XML_ATTRIBUTE_NODE', '4.4', [1198], '5.0'],
            ['XML_TEXT_NODE', '4.4', [1199], '5.0'],
            ['XML_CDATA_SECTION_NODE', '4.4', [1200], '5.0'],
            ['XML_ENTITY_REF_NODE', '4.4', [1201], '5.0'],
            ['XML_ENTITY_NODE', '4.4', [1202], '5.0'],
            ['XML_PI_NODE', '4.4', [1203], '5.0'],
            ['XML_COMMENT_NODE', '4.4', [1204], '5.0'],
            ['XML_DOCUMENT_NODE', '4.4', [1205], '5.0'],
            ['XML_DOCUMENT_TYPE_NODE', '4.4', [1206], '5.0'],
            ['XML_DOCUMENT_FRAG_NODE', '4.4', [1207], '5.0'],
            ['XML_NOTATION_NODE', '4.4', [1208], '5.0'],
            ['XML_HTML_DOCUMENT_NODE', '4.4', [1209], '5.0'],
            ['XML_DTD_NODE', '4.4', [1210], '5.0'],
            ['XML_ELEMENT_DECL_NODE', '4.4', [1211], '5.0'],
            ['XML_ATTRIBUTE_DECL_NODE', '4.4', [1212], '5.0'],
            ['XML_ENTITY_DECL_NODE', '4.4', [1213], '5.0'],
            ['XML_NAMESPACE_DECL_NODE', '4.4', [1214], '5.0'],
            ['XML_ATTRIBUTE_CDATA', '4.4', [1215], '5.0'],
            ['XML_ATTRIBUTE_ID', '4.4', [1216], '5.0'],
            ['XML_ATTRIBUTE_IDREF', '4.4', [1217], '5.0'],
            ['XML_ATTRIBUTE_IDREFS', '4.4', [1218], '5.0'],
            ['XML_ATTRIBUTE_ENTITY', '4.4', [1219], '5.0'],
            ['XML_ATTRIBUTE_NMTOKEN', '4.4', [1220], '5.0'],
            ['XML_ATTRIBUTE_NMTOKENS', '4.4', [1221], '5.0'],
            ['XML_ATTRIBUTE_ENUMERATION', '4.4', [1222], '5.0'],
            ['XML_ATTRIBUTE_NOTATION', '4.4', [1223], '5.0'],
            ['DOM_PHP_ERR', '4.4', [1224], '5.0'],
            ['DOM_INDEX_SIZE_ERR', '4.4', [1225], '5.0'],
            ['DOMSTRING_SIZE_ERR', '4.4', [1226], '5.0'],
            ['DOM_HIERARCHY_REQUEST_ERR', '4.4', [1227], '5.0'],
            ['DOM_WRONG_DOCUMENT_ERR', '4.4', [1228], '5.0'],
            ['DOM_INVALID_CHARACTER_ERR', '4.4', [1229], '5.0'],
            ['DOM_NO_DATA_ALLOWED_ERR', '4.4', [1230], '5.0'],
            ['DOM_NO_MODIFICATION_ALLOWED_ERR', '4.4', [1231], '5.0'],
            ['DOM_NOT_FOUND_ERR', '4.4', [1232], '5.0'],
            ['DOM_NOT_SUPPORTED_ERR', '4.4', [1233], '5.0'],
            ['DOM_INUSE_ATTRIBUTE_ERR', '4.4', [1234], '5.0'],
            ['DOM_INVALID_STATE_ERR', '4.4', [1235], '5.0'],
            ['DOM_SYNTAX_ERR', '4.4', [1236], '5.0'],
            ['DOM_INVALID_MODIFICATION_ERR', '4.4', [1237], '5.0'],
            ['DOM_NAMESPACE_ERR', '4.4', [1238], '5.0'],
            ['DOM_INVALID_ACCESS_ERR', '4.4', [1239], '5.0'],
            ['DOM_VALIDATION_ERR', '4.4', [1240], '5.0'],
            ['XSL_CLONE_AUTO', '4.4', [1261], '5.0'],
            ['XSL_CLONE_NEVER', '4.4', [1262], '5.0'],
            ['XSL_CLONE_ALWAYS', '4.4', [1263], '5.0'],
            ['XSL_SECPREF_NONE', '4.4', [1264], '5.0'],
            ['XSL_SECPREF_READ_FILE', '4.4', [1265], '5.0'],
            ['XSL_SECPREF_WRITE_FILE', '4.4', [1266], '5.0'],
            ['XSL_SECPREF_CREATE_DIRECTORY', '4.4', [1267], '5.0'],
            ['XSL_SECPREF_READ_NETWORK', '4.4', [1268], '5.0'],
            ['XSL_SECPREF_WRITE_NETWORK', '4.4', [1269], '5.0'],
            ['XSL_SECPREF_DEFAULT', '4.4', [1270], '5.0'],
            ['SQLITE_ASSOC', '4.4', [1271], '5.0'],
            ['SQLITE_BOTH', '4.4', [1272], '5.0'],
            ['SQLITE_NUM', '4.4', [1273], '5.0'],
            ['SQLITE_OK', '4.4', [1274], '5.0'],
            ['SQLITE_ERROR', '4.4', [1275], '5.0'],
            ['SQLITE_INTERNAL', '4.4', [1276], '5.0'],
            ['SQLITE_PERM', '4.4', [1277], '5.0'],
            ['SQLITE_ABORT', '4.4', [1278], '5.0'],
            ['SQLITE_BUSY', '4.4', [1279], '5.0'],
            ['SQLITE_LOCKED', '4.4', [1280], '5.0'],
            ['SQLITE_NOMEM', '4.4', [1281], '5.0'],
            ['SQLITE_READONLY', '4.4', [1282], '5.0'],
            ['SQLITE_INTERRUPT', '4.4', [1283], '5.0'],
            ['SQLITE_IOERR', '4.4', [1284], '5.0'],
            ['SQLITE_NOTADB', '4.4', [1285], '5.0'],
            ['SQLITE_CORRUPT', '4.4', [1286], '5.0'],
            ['SQLITE_FORMAT', '4.4', [1287], '5.0'],
            ['SQLITE_NOTFOUND', '4.4', [1288], '5.0'],
            ['SQLITE_FULL', '4.4', [1289], '5.0'],
            ['SQLITE_CANTOPEN', '4.4', [1290], '5.0'],
            ['SQLITE_PROTOCOL', '4.4', [1291], '5.0'],
            ['SQLITE_EMPTY', '4.4', [1292], '5.0'],
            ['SQLITE_SCHEMA', '4.4', [1293], '5.0'],
            ['SQLITE_TOOBIG', '4.4', [1294], '5.0'],
            ['SQLITE_CONSTRAINT', '4.4', [1295], '5.0'],
            ['SQLITE_MISMATCH', '4.4', [1296], '5.0'],
            ['SQLITE_MISUSE', '4.4', [1297], '5.0'],
            ['SQLITE_NOLFS', '4.4', [1298], '5.0'],
            ['SQLITE_AUTH', '4.4', [1299], '5.0'],
            ['SQLITE_ROW', '4.4', [1300], '5.0'],
            ['SQLITE_DONE', '4.4', [1301], '5.0'],
            ['MYSQLI_READ_DEFAULT_GROUP', '4.4', [1303], '5.0'],
            ['MYSQLI_READ_DEFAULT_FILE', '4.4', [1304], '5.0'],
            ['MYSQLI_OPT_CONNECT_TIMEOUT', '4.4', [1305], '5.0'],
            ['MYSQLI_OPT_LOCAL_INFILE', '4.4', [1306], '5.0'],
            ['MYSQLI_INIT_COMMAND', '4.4', [1307], '5.0'],
            ['MYSQLI_CLIENT_SSL', '4.4', [1308], '5.0'],
            ['MYSQLI_CLIENT_COMPRESS', '4.4', [1309], '5.0'],
            ['MYSQLI_CLIENT_INTERACTIVE', '4.4', [1310], '5.0'],
            ['MYSQLI_CLIENT_IGNORE_SPACE', '4.4', [1311], '5.0'],
            ['MYSQLI_CLIENT_NO_SCHEMA', '4.4', [1312], '5.0'],
            ['MYSQLI_CLIENT_MULTI_QUERIES', '4.4', [1313], '5.0'],
            ['MYSQLI_STORE_RESULT', '4.4', [1314], '5.0'],
            ['MYSQLI_USE_RESULT', '4.4', [1315], '5.0'],
            ['MYSQLI_ASSOC', '4.4', [1316], '5.0'],
            ['MYSQLI_NUM', '4.4', [1317], '5.0'],
            ['MYSQLI_BOTH', '4.4', [1318], '5.0'],
            ['MYSQLI_NOT_NULL_FLAG', '4.4', [1319], '5.0'],
            ['MYSQLI_PRI_KEY_FLAG', '4.4', [1320], '5.0'],
            ['MYSQLI_UNIQUE_KEY_FLAG', '4.4', [1321], '5.0'],
            ['MYSQLI_MULTIPLE_KEY_FLAG', '4.4', [1322], '5.0'],
            ['MYSQLI_BLOB_FLAG', '4.4', [1323], '5.0'],
            ['MYSQLI_UNSIGNED_FLAG', '4.4', [1324], '5.0'],
            ['MYSQLI_ZEROFILL_FLAG', '4.4', [1325], '5.0'],
            ['MYSQLI_AUTO_INCREMENT_FLAG', '4.4', [1326], '5.0'],
            ['MYSQLI_TIMESTAMP_FLAG', '4.4', [1327], '5.0'],
            ['MYSQLI_SET_FLAG', '4.4', [1328], '5.0'],
            ['MYSQLI_NUM_FLAG', '4.4', [1329], '5.0'],
            ['MYSQLI_PART_KEY_FLAG', '4.4', [1330], '5.0'],
            ['MYSQLI_GROUP_FLAG', '4.4', [1331], '5.0'],
            ['MYSQLI_TYPE_DECIMAL', '4.4', [1332], '5.0'],
            ['MYSQLI_TYPE_NEWDECIMAL', '4.4', [1333], '5.0'],
            ['MYSQLI_TYPE_BIT', '4.4', [1334], '5.0'],
            ['MYSQLI_TYPE_TINY', '4.4', [1335], '5.0'],
            ['MYSQLI_TYPE_SHORT', '4.4', [1336], '5.0'],
            ['MYSQLI_TYPE_LONG', '4.4', [1337], '5.0'],
            ['MYSQLI_TYPE_FLOAT', '4.4', [1338], '5.0'],
            ['MYSQLI_TYPE_DOUBLE', '4.4', [1339], '5.0'],
            ['MYSQLI_TYPE_NULL', '4.4', [1340], '5.0'],
            ['MYSQLI_TYPE_TIMESTAMP', '4.4', [1341], '5.0'],
            ['MYSQLI_TYPE_LONGLONG', '4.4', [1342], '5.0'],
            ['MYSQLI_TYPE_INT24', '4.4', [1343], '5.0'],
            ['MYSQLI_TYPE_DATE', '4.4', [1344], '5.0'],
            ['MYSQLI_TYPE_TIME', '4.4', [1345], '5.0'],
            ['MYSQLI_TYPE_DATETIME', '4.4', [1346], '5.0'],
            ['MYSQLI_TYPE_YEAR', '4.4', [1347], '5.0'],
            ['MYSQLI_TYPE_NEWDATE', '4.4', [1348], '5.0'],
            ['MYSQLI_TYPE_INTERVAL', '4.4', [1349], '5.0'],
            ['MYSQLI_TYPE_ENUM', '4.4', [1350], '5.0'],
            ['MYSQLI_TYPE_SET', '4.4', [1351], '5.0'],
            ['MYSQLI_TYPE_TINY_BLOB', '4.4', [1352], '5.0'],
            ['MYSQLI_TYPE_MEDIUM_BLOB', '4.4', [1353], '5.0'],
            ['MYSQLI_TYPE_LONG_BLOB', '4.4', [1354], '5.0'],
            ['MYSQLI_TYPE_BLOB', '4.4', [1355], '5.0'],
            ['MYSQLI_TYPE_VAR_STRING', '4.4', [1356], '5.0'],
            ['MYSQLI_TYPE_STRING', '4.4', [1357], '5.0'],
            ['MYSQLI_TYPE_CHAR', '4.4', [1358], '5.0'],
            ['MYSQLI_TYPE_GEOMETRY', '4.4', [1359], '5.0'],
            ['MYSQLI_NEED_DATA', '4.4', [1360], '5.0'],
            ['MYSQLI_NO_DATA', '4.4', [1361], '5.0'],
            ['MYSQLI_CURSOR_TYPE_FOR_UPDATE', '4.4', [1362], '5.0'],
            ['MYSQLI_CURSOR_TYPE_NO_CURSOR', '4.4', [1363], '5.0'],
            ['MYSQLI_CURSOR_TYPE_READ_ONLY', '4.4', [1364], '5.0'],
            ['MYSQLI_CURSOR_TYPE_SCROLLABLE', '4.4', [1365], '5.0'],
            ['MYSQLI_STMT_ATTR_CURSOR_TYPE', '4.4', [1366], '5.0'],
            ['MYSQLI_STMT_ATTR_PREFETCH_ROWS', '4.4', [1367], '5.0'],
            ['MYSQLI_STMT_ATTR_UPDATE_MAX_LENGTH', '4.4', [1368], '5.0'],
            ['MYSQLI_SET_CHARSET_NAME', '4.4', [1369], '5.0'],
            ['MYSQLI_REPORT_INDEX', '4.4', [1370], '5.0'],
            ['MYSQLI_REPORT_ERROR', '4.4', [1371], '5.0'],
            ['MYSQLI_REPORT_STRICT', '4.4', [1372], '5.0'],
            ['MYSQLI_REPORT_ALL', '4.4', [1373], '5.0'],
            ['MYSQLI_REPORT_OFF', '4.4', [1374], '5.0'],
            ['MYSQLI_DEBUG_TRACE_ENABLED', '4.4', [1375], '5.0'],
            ['MYSQLI_SERVER_QUERY_NO_GOOD_INDEX_USED', '4.4', [1376], '5.0'],
            ['MYSQLI_SERVER_QUERY_NO_INDEX_USED', '4.4', [1377], '5.0'],
            ['MYSQLI_REFRESH_GRANT', '4.4', [1378], '5.0'],
            ['MYSQLI_REFRESH_LOG', '4.4', [1379], '5.0'],
            ['MYSQLI_REFRESH_TABLES', '4.4', [1380], '5.0'],
            ['MYSQLI_REFRESH_HOSTS', '4.4', [1381], '5.0'],
            ['MYSQLI_REFRESH_STATUS', '4.4', [1382], '5.0'],
            ['MYSQLI_REFRESH_THREADS', '4.4', [1383], '5.0'],
            ['MYSQLI_REFRESH_SLAVE', '4.4', [1384], '5.0'],
            ['MYSQLI_REFRESH_MASTER', '4.4', [1385], '5.0'],
            ['MYSQLI_TRANS_COR_AND_CHAIN', '4.4', [1386], '5.0'],
            ['MYSQLI_TRANS_COR_AND_NO_CHAIN', '4.4', [1387], '5.0'],
            ['MYSQLI_TRANS_COR_RELEASE', '4.4', [1388], '5.0'],
            ['MYSQLI_TRANS_COR_NO_RELEASE', '4.4', [1389], '5.0'],
            ['MYSQLI_TRANS_START_READ_ONLY', '4.4', [1390], '5.0'],
            ['MYSQLI_TRANS_START_READ_WRITE', '4.4', [1391], '5.0'],
            ['MYSQLI_TRANS_START_CONSISTENT_SNAPSHOT', '4.4', [1392], '5.0'],

            ['SORT_LOCALE_STRING', '5.0.1', [25], '5.1', '5.0'],
            ['PHP_EOL', '5.0.1', [26], '5.1', '5.0'],
            ['PHP_INT_MAX', '5.0.4', [27], '5.1', '5.0'],
            ['PHP_INT_SIZE', '5.0.4', [28], '5.1', '5.0'],
            ['__COMPILER_HALT_OFFSET__', '5.0', [29], '5.1'],
            ['GLOB_ERR', '5.0', [30], '5.1'],
            ['CURLFTPAUTH_DEFAULT', '5.0', [99], '5.1'],
            ['CURLFTPAUTH_SSL', '5.0', [100], '5.1'],
            ['CURLFTPAUTH_TLS', '5.0', [101], '5.1'],
            ['CURLOPT_AUTOREFERER', '5.0', [31], '5.1'],
            ['CURLOPT_BINARYTRANSFER', '5.0', [32], '5.1'],
            ['CURLOPT_COOKIESESSION', '5.0', [33], '5.1'],
            ['CURLOPT_FTPSSLAUTH', '5.0', [34], '5.1'],
            ['CURLOPT_PROXYAUTH', '5.0', [35], '5.1'],
            ['CURLOPT_TIMECONDITION', '5.0', [36], '5.1'],
            ['POSIX_F_OK', '5.0', [37], '5.1'],
            ['POSIX_R_OK', '5.0', [38], '5.1'],
            ['POSIX_W_OK', '5.0', [39], '5.1'],
            ['POSIX_X_OK', '5.0', [40], '5.1'],
            ['POSIX_S_IFBLK', '5.0', [41], '5.1'],
            ['POSIX_S_IFCHR', '5.0', [42], '5.1'],
            ['POSIX_S_IFIFO', '5.0', [43], '5.1'],
            ['POSIX_S_IFREG', '5.0', [44], '5.1'],
            ['POSIX_S_IFSOCK', '5.0', [45], '5.1'],
            ['STREAM_IPPROTO_ICMP', '5.0', [46], '5.1'],
            ['STREAM_IPPROTO_IP', '5.0', [47], '5.1'],
            ['STREAM_IPPROTO_RAW', '5.0', [48], '5.1'],
            ['STREAM_IPPROTO_TCP', '5.0', [49], '5.1'],
            ['STREAM_IPPROTO_UDP', '5.0', [50], '5.1'],
            ['STREAM_PF_INET', '5.0', [51], '5.1'],
            ['STREAM_PF_INET6', '5.0', [52], '5.1'],
            ['STREAM_PF_UNIX', '5.0', [53], '5.1'],
            ['STREAM_SOCK_DGRAM', '5.0', [54], '5.1'],
            ['STREAM_SOCK_RAW', '5.0', [55], '5.1'],
            ['STREAM_SOCK_RDM', '5.0', [56], '5.1'],
            ['STREAM_SOCK_SEQPACKET', '5.0', [57], '5.1'],
            ['STREAM_SOCK_STREAM', '5.0', [58], '5.1'],
            ['T_HALT_COMPILER', '5.0', [59], '5.1'],
            ['LIBXML_COMPACT', '5.0', [1241], '5.1'],
            ['LIBXML_DOTTED_VERSION', '5.0', [1242], '5.1'],
            ['LIBXML_DTDATTR', '5.0', [1243], '5.1'],
            ['LIBXML_DTDLOAD', '5.0', [1244], '5.1'],
            ['LIBXML_DTDVALID', '5.0', [1245], '5.1'],
            ['LIBXML_ERR_ERROR', '5.0', [1246], '5.1'],
            ['LIBXML_ERR_FATAL', '5.0', [1247], '5.1'],
            ['LIBXML_ERR_NONE', '5.0', [1248], '5.1'],
            ['LIBXML_ERR_WARNING', '5.0', [1249], '5.1'],
            ['LIBXML_NOBLANKS', '5.0', [1250], '5.1'],
            ['LIBXML_NOCDATA', '5.0', [1251], '5.1'],
            ['LIBXML_NOEMPTYTAG', '5.0', [1252], '5.1'],
            ['LIBXML_NOENT', '5.0', [1253], '5.1'],
            ['LIBXML_NOERROR', '5.0', [1254], '5.1'],
            ['LIBXML_NONET', '5.0', [1255], '5.1'],
            ['LIBXML_NOWARNING', '5.0', [1256], '5.1'],
            ['LIBXML_NOXMLDECL', '5.0', [1257], '5.1'],
            ['LIBXML_NSCLEAN', '5.0', [1258], '5.1'],
            ['LIBXML_VERSION', '5.0', [1259], '5.1'],
            ['LIBXML_XINCLUDE', '5.0', [1260], '5.1'],
            ['MYSQLI_DATA_TRUNCATED', '5.0', [1393], '5.1'],

            ['DATE_ATOM', '5.1.0', [60], '5.2', '5.1'],
            ['DATE_COOKIE', '5.1.0', [61], '5.2', '5.1'],
            ['DATE_ISO8601', '5.1.0', [62], '5.2', '5.1'],
            ['DATE_RFC822', '5.1.0', [63], '5.2', '5.1'],
            ['DATE_RFC850', '5.1.0', [64], '5.2', '5.1'],
            ['DATE_RFC1036', '5.1.0', [65], '5.2', '5.1'],
            ['DATE_RFC1123', '5.1.0', [66], '5.2', '5.1'],
            ['DATE_RFC2822', '5.1.0', [67], '5.2', '5.1'],
            ['DATE_RFC3339', '5.1.0', [68], '5.2', '5.1'],
            ['DATE_RSS', '5.1.0', [69], '5.2', '5.1'],
            ['DATE_W3C', '5.1.0', [70], '5.2', '5.1'],
            ['SUNFUNCS_RET_TIMESTAMP', '5.1.1', [71], '5.2', '5.1'],
            ['SUNFUNCS_RET_STRING', '5.1.1', [72], '5.2', '5.1'],
            ['SUNFUNCS_RET_DOUBLE', '5.1.1', [73], '5.2', '5.1'],
            ['HASH_HMAC', '5.1.1', [883], '5.2', '5.1'],
            ['LIBXSLT_VERSION', '5.1.1', [74], '5.2', '5.1'],
            ['LIBXSLT_DOTTED_VERSION', '5.1.1', [75], '5.2', '5.1'],
            ['LIBEXSLT_VERSION', '5.1.1', [76], '5.2', '5.1'],
            ['LIBEXSLT_DOTTED_VERSION', '5.1.1', [77], '5.2', '5.1'],
            ['PHP_URL_SCHEME', '5.1.1', [78], '5.2', '5.1'],
            ['PHP_URL_HOST', '5.1.1', [79], '5.2', '5.1'],
            ['PHP_URL_PORT', '5.1.1', [80], '5.2', '5.1'],
            ['PHP_URL_USER', '5.1.1', [81], '5.2', '5.1'],
            ['PHP_URL_PASS', '5.1.1', [82], '5.2', '5.1'],
            ['PHP_URL_PATH', '5.1.1', [83], '5.2', '5.1'],
            ['PHP_URL_QUERY', '5.1.1', [84], '5.2', '5.1'],
            ['PHP_URL_FRAGMENT', '5.1.1', [85], '5.2', '5.1'],
            ['PHP_QUERY_RFC1738', '5.1.1', [86], '5.2', '5.1'],
            ['PHP_QUERY_RFC3986', '5.1.1', [87], '5.2', '5.1'],
            ['CURLINFO_HEADER_OUT', '5.1.2', [88], '5.2', '5.1'],
            ['E_RECOVERABLE_ERROR', '5.1', [89], '5.2'],
            ['M_EULER', '5.1', [90], '5.2'],
            ['M_LNPI', '5.1', [91], '5.2'],
            ['M_SQRT3', '5.1', [92], '5.2'],
            ['M_SQRTPI', '5.1', [93], '5.2'],
            ['PATHINFO_FILENAME', '5.1', [94], '5.2'],
            ['UPLOAD_ERR_EXTENSION', '5.1', [95], '5.2'],
            ['CURLE_FILESIZE_EXCEEDED', '5.1', [96], '5.2'],
            ['CURLE_FTP_SSL_FAILED', '5.1', [97], '5.2'],
            ['CURLE_LDAP_INVALID_URL', '5.1', [98], '5.2'],
            ['CURLFTPSSL_ALL', '5.1', [102], '5.2'],
            ['CURLFTPSSL_CONTROL', '5.1', [103], '5.2'],
            ['CURLFTPSSL_NONE', '5.1', [104], '5.2'],
            ['CURLFTPSSL_TRY', '5.1', [105], '5.2'],
            ['CURLOPT_FTP_SSL', '5.1', [106], '5.2'],
            ['SWFTEXTFIELD_USEFONT', '5.1', [107], '5.2'],
            ['SWFTEXTFIELD_AUTOSIZE', '5.1', [108], '5.2'],
            ['SWF_SOUND_NOT_COMPRESSED', '5.1', [109], '5.2'],
            ['SWF_SOUND_ADPCM_COMPRESSED', '5.1', [110], '5.2'],
            ['SWF_SOUND_MP3_COMPRESSED', '5.1', [111], '5.2'],
            ['SWF_SOUND_NOT_COMPRESSED_LE', '5.1', [112], '5.2'],
            ['SWF_SOUND_NELLY_COMPRESSED', '5.1', [113], '5.2'],
            ['SWF_SOUND_5KHZ', '5.1', [114], '5.2'],
            ['SWF_SOUND_11KHZ', '5.1', [115], '5.2'],
            ['SWF_SOUND_22KHZ', '5.1', [116], '5.2'],
            ['SWF_SOUND_44KHZ', '5.1', [117], '5.2'],
            ['SWF_SOUND_8BITS', '5.1', [118], '5.2'],
            ['SWF_SOUND_16BITS', '5.1', [119], '5.2'],
            ['SWF_SOUND_MONO', '5.1', [120], '5.2'],
            ['SWF_SOUND_STEREO', '5.1', [121], '5.2'],
            ['OPENSSL_KEYTYPE_EC', '5.1', [122], '5.2'],
            ['OPENSSL_VERSION_NUMBER', '5.1', [123], '5.2'],
            ['OPENSSL_VERSION_TEXT', '5.1', [124], '5.2'],
            ['PREG_BACKTRACK_LIMIT_ERROR', '5.1', [125], '5.2'],
            ['PREG_BAD_UTF8_ERROR', '5.1', [126], '5.2'],
            ['PREG_INTERNAL_ERROR', '5.1', [127], '5.2'],
            ['PREG_NO_ERROR', '5.1', [128], '5.2'],
            ['PREG_RECURSION_LIMIT_ERROR', '5.1', [129], '5.2'],
            ['SNMP_OID_OUTPUT_FULL', '5.1', [130], '5.2'],
            ['SNMP_OID_OUTPUT_NUMERIC', '5.1', [131], '5.2'],
            ['MSG_EAGAIN', '5.1', [132], '5.2'],
            ['MSG_ENOMSG', '5.1', [133], '5.2'],
            ['INPUT_POST', '5.1', [1063], '5.2'],
            ['INPUT_GET', '5.1', [1064], '5.2'],
            ['INPUT_COOKIE', '5.1', [1065], '5.2'],
            ['INPUT_ENV', '5.1', [1066], '5.2'],
            ['INPUT_SERVER', '5.1', [1067], '5.2'],
            ['INPUT_SESSION', '5.1', [1068], '5.2'],
            ['INPUT_REQUEST', '5.1', [1069], '5.2'],
            ['FILTER_FLAG_NONE', '5.1', [1070], '5.2'],
            ['FILTER_REQUIRE_SCALAR', '5.1', [1071], '5.2'],
            ['FILTER_REQUIRE_ARRAY', '5.1', [1072], '5.2'],
            ['FILTER_FORCE_ARRAY', '5.1', [1073], '5.2'],
            ['FILTER_NULL_ON_FAILURE', '5.1', [1074], '5.2'],
            ['FILTER_VALIDATE_INT', '5.1', [1075], '5.2'],
            ['FILTER_VALIDATE_BOOLEAN', '5.1', [1076], '5.2'],
            ['FILTER_VALIDATE_FLOAT', '5.1', [1077], '5.2'],
            ['FILTER_VALIDATE_REGEXP', '5.1', [1078], '5.2'],
            ['FILTER_VALIDATE_URL', '5.1', [1079], '5.2'],
            ['FILTER_VALIDATE_EMAIL', '5.1', [1080], '5.2'],
            ['FILTER_VALIDATE_IP', '5.1', [1081], '5.2'],
            ['FILTER_DEFAULT', '5.1', [1082], '5.2'],
            ['FILTER_UNSAFE_RAW', '5.1', [1083], '5.2'],
            ['FILTER_SANITIZE_STRING', '5.1', [1084], '5.2'],
            ['FILTER_SANITIZE_STRIPPED', '5.1', [1085], '5.2'],
            ['FILTER_SANITIZE_ENCODED', '5.1', [1086], '5.2'],
            ['FILTER_SANITIZE_SPECIAL_CHARS', '5.1', [1087], '5.2'],
            ['FILTER_SANITIZE_EMAIL', '5.1', [1088], '5.2'],
            ['FILTER_SANITIZE_URL', '5.1', [1089], '5.2'],
            ['FILTER_SANITIZE_NUMBER_INT', '5.1', [1090], '5.2'],
            ['FILTER_SANITIZE_NUMBER_FLOAT', '5.1', [1091], '5.2'],
            ['FILTER_SANITIZE_MAGIC_QUOTES', '5.1', [1092], '5.2'],
            ['FILTER_CALLBACK', '5.1', [1093], '5.2'],
            ['FILTER_FLAG_ALLOW_OCTAL', '5.1', [1094], '5.2'],
            ['FILTER_FLAG_ALLOW_HEX', '5.1', [1095], '5.2'],
            ['FILTER_FLAG_STRIP_LOW', '5.1', [1096], '5.2'],
            ['FILTER_FLAG_STRIP_HIGH', '5.1', [1097], '5.2'],
            ['FILTER_FLAG_ENCODE_LOW', '5.1', [1098], '5.2'],
            ['FILTER_FLAG_ENCODE_HIGH', '5.1', [1099], '5.2'],
            ['FILTER_FLAG_ENCODE_AMP', '5.1', [1100], '5.2'],
            ['FILTER_FLAG_NO_ENCODE_QUOTES', '5.1', [1101], '5.2'],
            ['FILTER_FLAG_EMPTY_STRING_NULL', '5.1', [1102], '5.2'],
            ['FILTER_FLAG_ALLOW_FRACTION', '5.1', [1103], '5.2'],
            ['FILTER_FLAG_ALLOW_THOUSAND', '5.1', [1104], '5.2'],
            ['FILTER_FLAG_ALLOW_SCIENTIFIC', '5.1', [1105], '5.2'],
            ['FILTER_FLAG_PATH_REQUIRED', '5.1', [1106], '5.2'],
            ['FILTER_FLAG_QUERY_REQUIRED', '5.1', [1107], '5.2'],
            ['FILTER_FLAG_SCHEME_REQUIRED', '5.1', [1108], '5.2'],
            ['FILTER_FLAG_HOST_REQUIRED', '5.1', [1109], '5.2'],
            ['FILTER_FLAG_IPV4', '5.1', [1110], '5.2'],
            ['FILTER_FLAG_IPV6', '5.1', [1111], '5.2'],
            ['FILTER_FLAG_NO_RES_RANGE', '5.1', [1112], '5.2'],
            ['FILTER_FLAG_NO_PRIV_RANGE', '5.1', [1113], '5.2'],

            ['CURLOPT_TCP_NODELAY', '5.2.0', [134], '5.3', '5.2'],
            ['STREAM_SHUT_RD', '5.2.0', [135], '5.3', '5.2'],
            ['STREAM_SHUT_WR', '5.2.0', [136], '5.3', '5.2'],
            ['STREAM_SHUT_RDWR', '5.2.0', [137], '5.3', '5.2'],
            ['GMP_VERSION', '5.2.1', [138], '5.3', '5.2'],
            ['CURLOPT_TIMEOUT_MS', '5.2.2', [139], '5.3', '5.2'],
            ['CURLOPT_CONNECTTIMEOUT_MS', '5.2.2', [140], '5.3', '5.2'],
            ['CURLOPT_PRIVATE', '5.2.3', [141], '5.3', '5.2'],
            ['CURLINFO_PRIVATE', '5.2.3', [142], '5.3', '5.2'],
            ['GD_VERSION', '5.2.3', [143], '5.3', '5.2'],
            ['GD_MAJOR_VERSION', '5.2.3', [144], '5.3', '5.2'],
            ['GD_MINOR_VERSION', '5.2.3', [145], '5.3', '5.2'],
            ['GD_RELEASE_VERSION', '5.2.3', [146], '5.3', '5.2'],
            ['GD_EXTRA_VERSION', '5.2.3', [147], '5.3', '5.2'],
            ['PCRE_VERSION', '5.2.3', [148], '5.3', '5.2'],
            ['PHP_MAJOR_VERSION', '5.2.6', [149], '5.3', '5.2'],
            ['PHP_MINOR_VERSION', '5.2.6', [150], '5.3', '5.2'],
            ['PHP_RELEASE_VERSION', '5.2.6', [151], '5.3', '5.2'],
            ['PHP_VERSION_ID', '5.2.6', [152], '5.3', '5.2'],
            ['PHP_EXTRA_VERSION', '5.2.6', [153], '5.3', '5.2'],
            ['PHP_ZTS', '5.2.6', [154], '5.3', '5.2'],
            ['PHP_DEBUG', '5.2.6', [155], '5.3', '5.2'],
            ['FILE_BINARY', '5.2.6', [156], '5.3', '5.2'],
            ['FILE_TEXT', '5.2.6', [157], '5.3', '5.2'],
            ['TCP_NODELAY', '5.2.6', [158], '5.3', '5.2'],
            ['CURLOPT_PROTOCOLS', '5.2.9', [159], '5.3', '5.2'],
            ['CURLOPT_REDIR_PROTOCOLS', '5.2.9', [160], '5.3', '5.2'],
            ['CURLPROXY_SOCKS4', '5.2.9', [161], '5.3', '5.2'],
            ['LIBXML_PARSEHUGE', '5.2.11', [162], '5.3', '5.2'],
            ['ENT_IGNORE', '5.2', [163], '5.3'],
            ['E_DEPRECATED', '5.2', [164], '5.3'],
            ['E_USER_DEPRECATED', '5.2', [165], '5.3'],
            ['INI_SCANNER_NORMAL', '5.2', [166], '5.3'],
            ['INI_SCANNER_RAW', '5.2', [167], '5.3'],
            ['PHP_MAXPATHLEN', '5.2', [168], '5.3'],
            ['PHP_WINDOWS_NT_DOMAIN_CONTROLLER', '5.2', [169], '5.3'],
            ['PHP_WINDOWS_NT_SERVER', '5.2', [170], '5.3'],
            ['PHP_WINDOWS_NT_WORKSTATION', '5.2', [171], '5.3'],
            ['PHP_WINDOWS_VERSION_BUILD', '5.2', [172], '5.3'],
            ['PHP_WINDOWS_VERSION_MAJOR', '5.2', [173], '5.3'],
            ['PHP_WINDOWS_VERSION_MINOR', '5.2', [174], '5.3'],
            ['PHP_WINDOWS_VERSION_PLATFORM', '5.2', [175], '5.3'],
            ['PHP_WINDOWS_VERSION_PRODUCTTYPE', '5.2', [176], '5.3'],
            ['PHP_WINDOWS_VERSION_SP_MAJOR', '5.2', [177], '5.3'],
            ['PHP_WINDOWS_VERSION_SP_MINOR', '5.2', [178], '5.3'],
            ['PHP_WINDOWS_VERSION_SUITEMASK', '5.2', [179], '5.3'],
            ['CURLINFO_CERTINFO', '5.2', [180], '5.3'],
            ['CURLOPT_PROGRESSFUNCTION', '5.2', [181], '5.3'],
            ['CURLE_SSH', '5.2', [182], '5.3'],
            ['ENCHANT_MYSPELL', '5.2', [894], '5.3'],
            ['ENCHANT_ISPELL', '5.2', [895], '5.3'],
            ['IMG_FILTER_PIXELATE', '5.2', [183], '5.3'],
            ['IMAGETYPE_ICO', '5.2', [184], '5.3'],
            ['FILEINFO_MIME_TYPE', '5.2', [185], '5.3'],
            ['FILEINFO_MIME_ENCODING', '5.2', [186], '5.3'],
            ['INTL_MAX_LOCALE_LEN', '5.2', [896], '5.3'],
            ['IDNA_DEFAULT', '5.2', [897], '5.3'],
            ['IDNA_ALLOW_UNASSIGNED', '5.2', [898], '5.3'],
            ['IDNA_USE_STD3_RULES', '5.2', [899], '5.3'],
            ['JSON_ERROR_CTRL_CHAR', '5.2', [187], '5.3'],
            ['JSON_ERROR_DEPTH', '5.2', [188], '5.3'],
            ['JSON_ERROR_NONE', '5.2', [189], '5.3'],
            ['JSON_ERROR_STATE_MISMATCH', '5.2', [190], '5.3'],
            ['JSON_ERROR_SYNTAX', '5.2', [191], '5.3'],
            ['JSON_FORCE_OBJECT', '5.2', [192], '5.3'],
            ['JSON_HEX_TAG', '5.2', [193], '5.3'],
            ['JSON_HEX_AMP', '5.2', [194], '5.3'],
            ['JSON_HEX_APOS', '5.2', [195], '5.3'],
            ['JSON_HEX_QUOT', '5.2', [196], '5.3'],
            ['LDAP_OPT_NETWORK_TIMEOUT', '5.2', [197], '5.3'],
            ['LIBXML_LOADED_VERSION', '5.2', [198], '5.3'],
            ['PHP_ROUND_HALF_UP', '5.2', [199], '5.3'],
            ['PHP_ROUND_HALF_DOWN', '5.2', [200], '5.3'],
            ['PHP_ROUND_HALF_EVEN', '5.2', [201], '5.3'],
            ['PHP_ROUND_HALF_ODD', '5.2', [202], '5.3'],
            ['MYSQLI_OPT_INT_AND_FLOAT_NATIVE', '5.2', [203], '5.3'],
            ['MYSQLI_OPT_NET_CMD_BUFFER_SIZE', '5.2', [204], '5.3'],
            ['MYSQLI_OPT_NET_READ_BUFFER_SIZE', '5.2', [205], '5.3'],
            ['MYSQLI_OPT_SSL_VERIFY_SERVER_CERT', '5.2', [206], '5.3'],
            ['OCI_CRED_EXT', '5.2', [207], '5.3'],
            ['PREG_BAD_UTF8_OFFSET_ERROR', '5.2', [208], '5.3'],
            ['BUS_ADRALN', '5.2', [209], '5.3'],
            ['BUS_ADRERR', '5.2', [210], '5.3'],
            ['BUS_OBJERR', '5.2', [211], '5.3'],
            ['CLD_CONTIUNED', '5.2', [212], '5.3'],
            ['CLD_DUMPED', '5.2', [213], '5.3'],
            ['CLD_EXITED', '5.2', [214], '5.3'],
            ['CLD_KILLED', '5.2', [215], '5.3'],
            ['CLD_STOPPED', '5.2', [216], '5.3'],
            ['CLD_TRAPPED', '5.2', [217], '5.3'],
            ['FPE_FLTDIV', '5.2', [218], '5.3'],
            ['FPE_FLTINV', '5.2', [219], '5.3'],
            ['FPE_FLTOVF', '5.2', [220], '5.3'],
            ['FPE_FLTRES', '5.2', [221], '5.3'],
            ['FPE_FLTSUB', '5.2', [222], '5.3'],
            ['FPE_FLTUND', '5.2', [223], '5.3'],
            ['FPE_INTDIV', '5.2', [224], '5.3'],
            ['FPE_INTOVF', '5.2', [225], '5.3'],
            ['ILL_BADSTK', '5.2', [226], '5.3'],
            ['ILL_COPROC', '5.2', [227], '5.3'],
            ['ILL_ILLADR', '5.2', [228], '5.3'],
            ['ILL_ILLOPC', '5.2', [229], '5.3'],
            ['ILL_ILLOPN', '5.2', [230], '5.3'],
            ['ILL_ILLTRP', '5.2', [231], '5.3'],
            ['ILL_PRVOPC', '5.2', [232], '5.3'],
            ['ILL_PRVREG', '5.2', [233], '5.3'],
            ['POLL_ERR', '5.2', [234], '5.3'],
            ['POLL_HUP', '5.2', [235], '5.3'],
            ['POLL_IN', '5.2', [236], '5.3'],
            ['POLL_MSG', '5.2', [237], '5.3'],
            ['POLL_OUT', '5.2', [238], '5.3'],
            ['POLL_PRI', '5.2', [239], '5.3'],
            ['SEGV_ACCERR', '5.2', [240], '5.3'],
            ['SEGV_MAPERR', '5.2', [241], '5.3'],
            ['SI_ASYNCIO', '5.2', [242], '5.3'],
            ['SI_KERNEL', '5.2', [243], '5.3'],
            ['SI_MSGGQ', '5.2', [244], '5.3'],
            ['SI_NOINFO', '5.2', [245], '5.3'],
            ['SI_QUEUE', '5.2', [246], '5.3'],
            ['SI_SIGIO', '5.2', [247], '5.3'],
            ['SI_TIMER', '5.2', [248], '5.3'],
            ['SI_TKILL', '5.2', [249], '5.3'],
            ['SI_USER', '5.2', [250], '5.3'],
            ['SIG_BLOCK', '5.2', [251], '5.3'],
            ['SIG_SETMASK', '5.2', [252], '5.3'],
            ['SIG_UNBLOCK', '5.2', [253], '5.3'],
            ['TRAP_BRKPT', '5.2', [254], '5.3'],
            ['TRAP_TRACE', '5.2', [255], '5.3'],
            ['T_DIR', '5.2', [256], '5.3'],
            ['T_GOTO', '5.2', [257], '5.3'],
            ['T_NAMESPACE', '5.2', [258], '5.3'],
            ['T_NS_C', '5.2', [259], '5.3'],
            ['T_NS_SEPARATOR', '5.2', [260], '5.3'],
            ['T_USE', '5.2', [261], '5.3'],
            ['FILEINFO_NONE', '5.2', [886], '5.3'],
            ['FILEINFO_SYMLINK', '5.2', [887], '5.3'],
            ['FILEINFO_MIME', '5.2', [888], '5.3'],
            ['FILEINFO_COMPRESS', '5.2', [889], '5.3'],
            ['FILEINFO_DEVICES', '5.2', [890], '5.3'],
            ['FILEINFO_CONTINUE', '5.2', [891], '5.3'],
            ['FILEINFO_PRESERVE_ATIME', '5.2', [892], '5.3'],
            ['FILEINFO_RAW', '5.2', [893], '5.3'],
            ['MYSQLI_ENUM_FLAG', '5.2', [1394], '5.3'],
            ['MYSQLI_BINARY_FLAG', '5.2', [1395], '5.3'],

            ['OCI_NO_AUTO_COMMIT', '5.3.1', [262], '5.4', '5.3'],
            ['OPENSSL_TLSEXT_SERVER_NAME', '5.3.1', [263], '5.4', '5.3'],
            ['FILTER_FLAG_STRIP_BACKTICK', '5.3.1', [1062], '5.4', '5.3'],
            ['JSON_ERROR_UTF8', '5.3.2', [264], '5.4', '5.3'],
            ['JSON_NUMERIC_CHECK', '5.3.2', [265], '5.4', '5.3'],
            ['DEBUG_BACKTRACE_IGNORE_ARGS', '5.3.5', [266], '5.4', '5.3'],
            ['CURLINFO_REDIRECT_URL', '5.3.6', [267], '5.4', '5.3'],
            ['PHP_MANDIR', '5.3.6', [268], '5.4', '5.3'],

            ['PHP_BINARY', '5.3', [269], '5.4'],
            ['SORT_NATURAL', '5.3', [270], '5.4'],
            ['SORT_FLAG_CASE', '5.3', [271], '5.4'],
            ['ENT_HTML401', '5.3', [272], '5.4'],
            ['ENT_XML1', '5.3', [273], '5.4'],
            ['ENT_XHTML', '5.3', [274], '5.4'],
            ['ENT_HTML5', '5.3', [275], '5.4'],
            ['ENT_SUBSTITUTE', '5.3', [276], '5.4'],
            ['ENT_DISALLOWED', '5.3', [277], '5.4'],
            ['IPPROTO_IP', '5.3', [278], '5.4'],
            ['IPPROTO_IPV6', '5.3', [279], '5.4'],
            ['IPV6_MULTICAST_HOPS', '5.3', [280], '5.4'],
            ['IPV6_MULTICAST_IF', '5.3', [281], '5.4'],
            ['IPV6_MULTICAST_LOOP', '5.3', [282], '5.4'],
            ['IP_MULTICAST_IF', '5.3', [283], '5.4'],
            ['IP_MULTICAST_LOOP', '5.3', [284], '5.4'],
            ['IP_MULTICAST_TTL', '5.3', [285], '5.4'],
            ['MCAST_JOIN_GROUP', '5.3', [286], '5.4'],
            ['MCAST_LEAVE_GROUP', '5.3', [287], '5.4'],
            ['MCAST_BLOCK_SOURCE', '5.3', [288], '5.4'],
            ['MCAST_UNBLOCK_SOURCE', '5.3', [289], '5.4'],
            ['MCAST_JOIN_SOURCE_GROUP', '5.3', [290], '5.4'],
            ['MCAST_LEAVE_SOURCE_GROUP', '5.3', [291], '5.4'],
            ['CURLOPT_MAX_RECV_SPEED_LARGE', '5.3', [292], '5.4'],
            ['CURLOPT_MAX_SEND_SPEED_LARGE', '5.3', [293], '5.4'],
            ['SCANDIR_SORT_ASCENDING', '5.3', [294], '5.4'],
            ['SCANDIR_SORT_DESCENDING', '5.3', [295], '5.4'],
            ['SCANDIR_SORT_NONE', '5.3', [296], '5.4'],
            ['LIBXML_HTML_NODEFDTD', '5.3', [297], '5.4'],
            ['LIBXML_HTML_NOIMPLIED', '5.3', [298], '5.4'],
            ['LIBXML_PEDANTIC', '5.3', [299], '5.4'],
            ['OPENSSL_CIPHER_AES_128_CBC', '5.3', [300], '5.4'],
            ['OPENSSL_CIPHER_AES_192_CBC', '5.3', [301], '5.4'],
            ['OPENSSL_CIPHER_AES_256_CBC', '5.3', [302], '5.4'],
            ['OPENSSL_RAW_DATA', '5.3', [303], '5.4'],
            ['OPENSSL_ZERO_PADDING', '5.3', [304], '5.4'],
            ['PHP_OUTPUT_HANDLER_CLEAN', '5.3', [305], '5.4'],
            ['PHP_OUTPUT_HANDLER_CLEANABLE', '5.3', [306], '5.4'],
            ['PHP_OUTPUT_HANDLER_DISABLED', '5.3', [307], '5.4'],
            ['PHP_OUTPUT_HANDLER_FINAL', '5.3', [308], '5.4'],
            ['PHP_OUTPUT_HANDLER_FLUSH', '5.3', [309], '5.4'],
            ['PHP_OUTPUT_HANDLER_FLUSHABLE', '5.3', [310], '5.4'],
            ['PHP_OUTPUT_HANDLER_REMOVABLE', '5.3', [311], '5.4'],
            ['PHP_OUTPUT_HANDLER_STARTED', '5.3', [312], '5.4'],
            ['PHP_OUTPUT_HANDLER_STDFLAGS', '5.3', [313], '5.4'],
            ['PHP_OUTPUT_HANDLER_WRITE', '5.3', [314], '5.4'],
            ['PHP_SESSION_ACTIVE', '5.3', [315], '5.4'],
            ['PHP_SESSION_DISABLED', '5.3', [316], '5.4'],
            ['PHP_SESSION_NONE', '5.3', [317], '5.4'],
            ['STREAM_META_ACCESS', '5.3', [318], '5.4'],
            ['STREAM_META_GROUP', '5.3', [319], '5.4'],
            ['STREAM_META_GROUP_NAME', '5.3', [320], '5.4'],
            ['STREAM_META_OWNER', '5.3', [321], '5.4'],
            ['STREAM_META_OWNER_NAME', '5.3', [322], '5.4'],
            ['STREAM_META_TOUCH', '5.3', [323], '5.4'],
            ['U_IDNA_DOMAIN_NAME_TOO_LONG_ERROR', '5.3', [324], '5.4'],
            ['IDNA_CHECK_BIDI', '5.3', [325], '5.4'],
            ['IDNA_CHECK_CONTEXTJ', '5.3', [326], '5.4'],
            ['IDNA_NONTRANSITIONAL_TO_ASCII', '5.3', [327], '5.4'],
            ['IDNA_NONTRANSITIONAL_TO_UNICODE', '5.3', [328], '5.4'],
            ['INTL_IDNA_VARIANT_2003', '5.3', [329], '5.4'],
            ['INTL_IDNA_VARIANT_UTS46', '5.3', [330], '5.4'],
            ['IDNA_ERROR_EMPTY_LABEL', '5.3', [331], '5.4'],
            ['IDNA_ERROR_LABEL_TOO_LONG', '5.3', [332], '5.4'],
            ['IDNA_ERROR_DOMAIN_NAME_TOO_LONG', '5.3', [333], '5.4'],
            ['IDNA_ERROR_LEADING_HYPHEN', '5.3', [334], '5.4'],
            ['IDNA_ERROR_TRAILING_HYPHEN', '5.3', [335], '5.4'],
            ['IDNA_ERROR_HYPHEN_3_4', '5.3', [336], '5.4'],
            ['IDNA_ERROR_LEADING_COMBINING_MARK', '5.3', [337], '5.4'],
            ['IDNA_ERROR_DISALLOWED', '5.3', [338], '5.4'],
            ['IDNA_ERROR_PUNYCODE', '5.3', [339], '5.4'],
            ['IDNA_ERROR_LABEL_HAS_DOT', '5.3', [340], '5.4'],
            ['IDNA_ERROR_INVALID_ACE_LABEL', '5.3', [341], '5.4'],
            ['IDNA_ERROR_BIDI', '5.3', [342], '5.4'],
            ['IDNA_ERROR_CONTEXTJ', '5.3', [343], '5.4'],
            ['JSON_PRETTY_PRINT', '5.3', [344], '5.4'],
            ['JSON_UNESCAPED_SLASHES', '5.3', [345], '5.4'],
            ['JSON_UNESCAPED_UNICODE', '5.3', [346], '5.4'],
            ['JSON_BIGINT_AS_STRING', '5.3', [347], '5.4'],
            ['JSON_OBJECT_AS_ARRAY', '5.3', [348], '5.4'],
            ['SNMP_OID_OUTPUT_SUFFIX', '5.3', [349], '5.4'],
            ['SNMP_OID_OUTPUT_MODULE', '5.3', [350], '5.4'],
            ['SNMP_OID_OUTPUT_UCD', '5.3', [351], '5.4'],
            ['SNMP_OID_OUTPUT_NONE', '5.3', [352], '5.4'],
            ['T_CALLABLE', '5.3', [1047], '5.4'],
            ['T_INSTEADOF', '5.3', [353], '5.4'],
            ['T_TRAIT', '5.3', [354], '5.4'],
            ['T_TRAIT_C', '5.3', [355], '5.4'],

            ['CURLINFO_PRIMARY_IP', '5.4.6', [356], '5.5', '5.4'],
            ['CURLINFO_PRIMARY_PORT', '5.4.6', [357], '5.5', '5.4'],
            ['CURLINFO_LOCAL_IP', '5.4.6', [358], '5.5', '5.4'],
            ['CURLINFO_LOCAL_PORT', '5.4.6', [359], '5.5', '5.4'],
            ['OPENSSL_ALGO_RMD160', '5.4.7', [360], '5.5', '5.4'],
            ['OPENSSL_ALGO_SHA224', '5.4.7', [361], '5.5', '5.4'],
            ['OPENSSL_ALGO_SHA256', '5.4.7', [362], '5.5', '5.4'],
            ['OPENSSL_ALGO_SHA384', '5.4.7', [363], '5.5', '5.4'],
            ['OPENSSL_ALGO_SHA512', '5.4.7', [364], '5.5', '5.4'],

            ['SO_REUSEPORT', '5.4.9', [1060], '5.5', '5.4'],

            ['FILTER_VALIDATE_MAC', '5.4', [365], '5.5'],
            ['IMG_AFFINE_TRANSLATE', '5.4', [366], '5.5'],
            ['IMG_AFFINE_SCALE', '5.4', [367], '5.5'],
            ['IMG_AFFINE_ROTATE', '5.4', [368], '5.5'],
            ['IMG_AFFINE_SHEAR_HORIZONTAL', '5.4', [369], '5.5'],
            ['IMG_AFFINE_SHEAR_VERTICAL', '5.4', [370], '5.5'],
            ['IMG_CROP_DEFAULT', '5.4', [371], '5.5'],
            ['IMG_CROP_TRANSPARENT', '5.4', [372], '5.5'],
            ['IMG_CROP_BLACK', '5.4', [373], '5.5'],
            ['IMG_CROP_WHITE', '5.4', [374], '5.5'],
            ['IMG_CROP_SIDES', '5.4', [375], '5.5'],
            ['IMG_FLIP_BOTH', '5.4', [376], '5.5'],
            ['IMG_FLIP_HORIZONTAL', '5.4', [377], '5.5'],
            ['IMG_FLIP_VERTICAL', '5.4', [378], '5.5'],
            ['IMG_BELL', '5.4', [379], '5.5'],
            ['IMG_BESSEL', '5.4', [380], '5.5'],
            ['IMG_BILINEAR_FIXED', '5.4', [381], '5.5'],
            ['IMG_BICUBIC', '5.4', [382], '5.5'],
            ['IMG_BICUBIC_FIXED', '5.4', [383], '5.5'],
            ['IMG_BLACKMAN', '5.4', [384], '5.5'],
            ['IMG_BOX', '5.4', [385], '5.5'],
            ['IMG_BSPLINE', '5.4', [386], '5.5'],
            ['IMG_CATMULLROM', '5.4', [387], '5.5'],
            ['IMG_GAUSSIAN', '5.4', [388], '5.5'],
            ['IMG_GENERALIZED_CUBIC', '5.4', [389], '5.5'],
            ['IMG_HERMITE', '5.4', [390], '5.5'],
            ['IMG_HAMMING', '5.4', [391], '5.5'],
            ['IMG_HANNING', '5.4', [392], '5.5'],
            ['IMG_MITCHELL', '5.4', [393], '5.5'],
            ['IMG_POWER', '5.4', [394], '5.5'],
            ['IMG_QUADRATIC', '5.4', [395], '5.5'],
            ['IMG_SINC', '5.4', [396], '5.5'],
            ['IMG_NEAREST_NEIGHBOUR', '5.4', [397], '5.5'],
            ['IMG_WEIGHTED4', '5.4', [398], '5.5'],
            ['IMG_TRIANGLE', '5.4', [399], '5.5'],
            ['JSON_ERROR_RECURSION', '5.4', [400], '5.5'],
            ['JSON_ERROR_INF_OR_NAN', '5.4', [401], '5.5'],
            ['JSON_ERROR_UNSUPPORTED_TYPE', '5.4', [402], '5.5'],
            ['JSON_PARTIAL_OUTPUT_ON_ERROR', '5.4', [403], '5.5'],
            ['MYSQLI_SERVER_PUBLIC_KEY', '5.4', [404], '5.5'],
            ['CURLOPT_SHARE', '5.4', [405], '5.5'],
            ['CURLOPT_SSL_OPTIONS', '5.4', [406], '5.5'],
            ['CURLOPT_COOKIELIST', '5.4', [1050], '5.5'],
            ['CURLOPT_TCP_KEEPALIVE', '5.4', [1053], '5.5'],
            ['CURLOPT_TCP_KEEPIDLE', '5.4', [1054], '5.5'],
            ['CURLOPT_TCP_KEEPINTVL', '5.4', [1055], '5.5'],
            ['CURLSSLOPT_ALLOW_BEAST', '5.4', [407], '5.5'],
            ['CURLOPT_USERNAME', '5.4', [408], '5.5'],
            ['CURLINFO_RESPONSE_CODE', '5.4', [409], '5.5'],
            ['CURLINFO_HTTP_CONNECTCODE', '5.4', [410], '5.5'],
            ['CURLINFO_HTTPAUTH_AVAIL', '5.4', [411], '5.5'],
            ['CURLINFO_PROXYAUTH_AVAIL', '5.4', [412], '5.5'],
            ['CURLINFO_OS_ERRNO', '5.4', [413], '5.5'],
            ['CURLINFO_NUM_CONNECTS', '5.4', [414], '5.5'],
            ['CURLINFO_SSL_ENGINES', '5.4', [415], '5.5'],
            ['CURLINFO_COOKIELIST', '5.4', [416], '5.5'],
            ['CURLINFO_FTP_ENTRY_PATH', '5.4', [417], '5.5'],
            ['CURLINFO_APPCONNECT_TIME', '5.4', [418], '5.5'],
            ['CURLINFO_CONDITION_UNMET', '5.4', [419], '5.5'],
            ['CURLINFO_RTSP_CLIENT_CSEQ', '5.4', [420], '5.5'],
            ['CURLINFO_RTSP_CSEQ_RECV', '5.4', [421], '5.5'],
            ['CURLINFO_RTSP_SERVER_CSEQ', '5.4', [422], '5.5'],
            ['CURLINFO_RTSP_SESSION_ID', '5.4', [423], '5.5'],
            ['CURLINFO_HTTP_CODE', '5.4', [1051], '5.5'],
            ['CURLMOPT_PIPELINING', '5.4', [424], '5.5'],
            ['CURLMOPT_MAXCONNECTS', '5.4', [425], '5.5'],
            ['CURLPAUSE_ALL', '5.4', [426], '5.5'],
            ['CURLPAUSE_CONT', '5.4', [427], '5.5'],
            ['CURLPAUSE_RECV', '5.4', [428], '5.5'],
            ['CURLPAUSE_RECV_CONT', '5.4', [429], '5.5'],
            ['CURLPAUSE_SEND', '5.4', [430], '5.5'],
            ['CURLPAUSE_SEND_CONT', '5.4', [431], '5.5'],
            ['SOAP_SSL_METHOD_TLS', '5.4', [432], '5.5'],
            ['SOAP_SSL_METHOD_SSLv2', '5.4', [433], '5.5'],
            ['SOAP_SSL_METHOD_SSLv3', '5.4', [434], '5.5'],
            ['SOAP_SSL_METHOD_SSLv23', '5.4', [435], '5.5'],
            ['T_FINALLY', '5.4', [436], '5.5'],
            ['T_YIELD', '5.4', [437], '5.5'],
            ['PASSWORD_BCRYPT', '5.4', [696], '5.5'],
            ['PASSWORD_DEFAULT', '5.4', [697], '5.5'],
            ['PASSWORD_BCRYPT_DEFAULT_COST', '5.4', [698], '5.5'],
            ['READLINE_LIB', '5.4', [882], '5.5'],
            ['LIBXML_SCHEMA_CREATE', '5.5.1', [438], '5.6', '5.5'],
            ['CURL_SSLVERSION_TLSv1_0', '5.5.18', [439], '5.6', '5.5'],
            ['CURL_SSLVERSION_TLSv1_1', '5.5.18', [440], '5.6', '5.5'],
            ['CURL_SSLVERSION_TLSv1_2', '5.5.18', [441], '5.6', '5.5'],
            ['CURLPROXY_SOCKS4A', '5.5.22', [442], '5.6', '5.5'],
            ['CURLPROXY_SOCKS5_HOSTNAME', '5.5.22', [443], '5.6', '5.5'],
            ['CURL_VERSION_HTTP2', '5.5.23', [444], '5.6', '5.5'],
            ['CURL_HTTP_VERSION_2_0', '5.5.23', [1052], '5.6', '5.5'],
            ['ARRAY_FILTER_USE_KEY', '5.5', [445], '5.6'],
            ['ARRAY_FILTER_USE_BOTH', '5.5', [446], '5.6'],
            ['LDAP_ESCAPE_DN', '5.5', [447], '5.6'],
            ['LDAP_ESCAPE_FILTER', '5.5', [448], '5.6'],
            ['OPENSSL_DEFAULT_STREAM_CIPHERS', '5.5', [449], '5.6'],
            ['STREAM_CRYPTO_METHOD_ANY_CLIENT', '5.5', [450], '5.6'],
            ['STREAM_CRYPTO_METHOD_ANY_SERVER', '5.5', [451], '5.6'],
            ['STREAM_CRYPTO_METHOD_TLSv1_0_CLIENT', '5.5', [452], '5.6'],
            ['STREAM_CRYPTO_METHOD_TLSv1_0_SERVER', '5.5', [453], '5.6'],
            ['STREAM_CRYPTO_METHOD_TLSv1_1_CLIENT', '5.5', [454], '5.6'],
            ['STREAM_CRYPTO_METHOD_TLSv1_1_SERVER', '5.5', [455], '5.6'],
            ['STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT', '5.5', [456], '5.6'],
            ['STREAM_CRYPTO_METHOD_TLSv1_2_SERVER', '5.5', [457], '5.6'],
            ['PGSQL_CONNECT_ASYNC', '5.5', [458], '5.6'],
            ['PGSQL_CONNECTION_AUTH_OK', '5.5', [459], '5.6'],
            ['PGSQL_CONNECTION_AWAITING_RESPONSE', '5.5', [460], '5.6'],
            ['PGSQL_CONNECTION_MADE', '5.5', [461], '5.6'],
            ['PGSQL_CONNECTION_SETENV', '5.5', [462], '5.6'],
            ['PGSQL_CONNECTION_SSL_STARTUP', '5.5', [463], '5.6'],
            ['PGSQL_CONNECTION_STARTED', '5.5', [464], '5.6'],
            ['PGSQL_DML_ESCAPE', '5.5', [465], '5.6'],
            ['PGSQL_POLLING_ACTIVE', '5.5', [466], '5.6'],
            ['PGSQL_POLLING_FAILED', '5.5', [467], '5.6'],
            ['PGSQL_POLLING_OK', '5.5', [468], '5.6'],
            ['PGSQL_POLLING_READING', '5.5', [469], '5.6'],
            ['PGSQL_POLLING_WRITING', '5.5', [470], '5.6'],
            ['T_ELLIPSIS', '5.5', [471], '5.6'],
            ['T_POW', '5.5', [472], '5.6'],
            ['T_POW_EQUAL', '5.5', [473], '5.6'],
            ['FTP_USEPASVADDRESS', '5.5', [1057], '5.6'],

            ['INI_SCANNER_TYPED', '5.6.0', [474], '7.0', '5.6'],
            ['JSON_PRESERVE_ZERO_FRACTION', '5.6.5', [475], '7.0', '5.6'],
            ['MYSQLI_CLIENT_SSL_DONT_VERIFY_SERVER_CERT', '5.6.15', [476], '7.0', '5.6'],
            ['IMG_WEBP', '5.6.24', [477], '7.0', '5.6'],

            ['TOKEN_PARSE', '5.6', [478], '7.0'],
            ['FILTER_VALIDATE_DOMAIN', '5.6', [479], '7.0'],
            ['FILTER_FLAG_HOSTNAME', '5.6', [1061], '7.0'],
            ['PHP_INT_MIN', '5.6', [480], '7.0'],
            ['CURLPIPE_NOTHING', '5.6', [481], '7.0'],
            ['CURLPIPE_HTTP1', '5.6', [482], '7.0'],
            ['CURLPIPE_MULTIPLEX', '5.6', [483], '7.0'],
            ['DATE_RFC3339_EXTENDED', '5.6', [885], '7.0'],
            ['JSON_ERROR_INVALID_PROPERTY_NAME', '5.6', [484], '7.0'],
            ['JSON_ERROR_UTF16', '5.6', [485], '7.0'],
            ['LIBXML_BIGLINES', '5.6', [486], '7.0'],
            ['PREG_JIT_STACKLIMIT_ERROR', '5.6', [487], '7.0'],
            ['POSIX_RLIMIT_AS', '5.6', [488], '7.0'],
            ['POSIX_RLIMIT_CORE', '5.6', [489], '7.0'],
            ['POSIX_RLIMIT_CPU', '5.6', [490], '7.0'],
            ['POSIX_RLIMIT_DATA', '5.6', [491], '7.0'],
            ['POSIX_RLIMIT_FSIZE', '5.6', [492], '7.0'],
            ['POSIX_RLIMIT_LOCKS', '5.6', [493], '7.0'],
            ['POSIX_RLIMIT_MEMLOCK', '5.6', [494], '7.0'],
            ['POSIX_RLIMIT_MSGQUEUE', '5.6', [495], '7.0'],
            ['POSIX_RLIMIT_NICE', '5.6', [496], '7.0'],
            ['POSIX_RLIMIT_NOFILE', '5.6', [497], '7.0'],
            ['POSIX_RLIMIT_NPROC', '5.6', [498], '7.0'],
            ['POSIX_RLIMIT_RSS', '5.6', [499], '7.0'],
            ['POSIX_RLIMIT_RTPRIO', '5.6', [500], '7.0'],
            ['POSIX_RLIMIT_RTTIME', '5.6', [501], '7.0'],
            ['POSIX_RLIMIT_SIGPENDING', '5.6', [502], '7.0'],
            ['POSIX_RLIMIT_STACK', '5.6', [503], '7.0'],
            ['POSIX_RLIMIT_INFINITY', '5.6', [504], '7.0'],
            ['T_COALESCE', '5.6', [822], '7.0'],
            ['T_SPACESHIP', '5.6', [505], '7.0'],
            ['T_YIELD_FROM', '5.6', [823], '7.0'],
            ['ZLIB_ENCODING_RAW', '5.6', [506], '7.0'],
            ['ZLIB_ENCODING_DEFLATE', '5.6', [507], '7.0'],
            ['ZLIB_ENCODING_GZIP', '5.6', [508], '7.0'],
            ['ZLIB_FILTERED', '5.6', [509], '7.0'],
            ['ZLIB_HUFFMAN_ONLY', '5.6', [510], '7.0'],
            ['ZLIB_FIXED', '5.6', [511], '7.0'],
            ['ZLIB_RLE', '5.6', [512], '7.0'],
            ['ZLIB_DEFAULT_STRATEGY', '5.6', [513], '7.0'],
            ['ZLIB_BLOCK', '5.6', [514], '7.0'],
            ['ZLIB_FINISH', '5.6', [515], '7.0'],
            ['ZLIB_FULL_FLUSH', '5.6', [516], '7.0'],
            ['ZLIB_NO_FLUSH', '5.6', [517], '7.0'],
            ['ZLIB_PARTIAL_FLUSH', '5.6', [518], '7.0'],
            ['ZLIB_SYNC_FLUSH', '5.6', [519], '7.0'],
            ['LDAP_OPT_X_TLS_REQUIRE_CERT', '5.6', [1058], '7.0'],
            ['VT_I8', '5.6', [1191], '7.0'],
            ['VT_UI8', '5.6', [1192], '7.0'],
            ['DISP_E_DIVBYZERO', '5.6', [1193], '7.0'],
            ['DISP_E_OVERFLOW', '5.6', [1194], '7.0'],
            ['DISP_E_BADINDEX', '5.6', [1195], '7.0'],
            ['MK_E_UNAVAILABLE', '5.6', [1196], '7.0'],

            ['CURL_HTTP_VERSION_2', '7.0.6', [520], '7.1', '7.0'],
            ['CURL_HTTP_VERSION_2_PRIOR_KNOWLEDGE', '7.0.6', [521], '7.1', '7.0'],
            ['CURL_HTTP_VERSION_2TLS', '7.0.6', [522], '7.1', '7.0'],
            ['CURL_REDIR_POST_301', '7.0.6', [523], '7.1', '7.0'],
            ['CURL_REDIR_POST_302', '7.0.6', [524], '7.1', '7.0'],
            ['CURL_REDIR_POST_303', '7.0.6', [525], '7.1', '7.0'],
            ['CURL_REDIR_POST_ALL', '7.0.6', [526], '7.1', '7.0'],
            ['CURL_VERSION_KERBEROS5', '7.0.6', [527], '7.1', '7.0'],
            ['CURL_VERSION_PSL', '7.0.6', [528], '7.1', '7.0'],
            ['CURL_VERSION_UNIX_SOCKETS', '7.0.6', [529], '7.1', '7.0'],
            ['CURLAUTH_NEGOTIATE', '7.0.6', [530], '7.1', '7.0'],
            ['CURLAUTH_NTLM_WB', '7.0.6', [531], '7.1', '7.0'],
            ['CURLFTP_CREATE_DIR', '7.0.6', [532], '7.1', '7.0'],
            ['CURLFTP_CREATE_DIR_NONE', '7.0.6', [533], '7.1', '7.0'],
            ['CURLFTP_CREATE_DIR_RETRY', '7.0.6', [534], '7.1', '7.0'],
            ['CURLHEADER_SEPARATE', '7.0.6', [535], '7.1', '7.0'],
            ['CURLHEADER_UNIFIED', '7.0.6', [536], '7.1', '7.0'],
            ['CURLMOPT_CHUNK_LENGTH_PENALTY_SIZE', '7.0.6', [537], '7.1', '7.0'],
            ['CURLMOPT_CONTENT_LENGTH_PENALTY_SIZE', '7.0.6', [538], '7.1', '7.0'],
            ['CURLMOPT_MAX_HOST_CONNECTIONS', '7.0.6', [539], '7.1', '7.0'],
            ['CURLMOPT_MAX_PIPELINE_LENGTH', '7.0.6', [540], '7.1', '7.0'],
            ['CURLMOPT_MAX_TOTAL_CONNECTIONS', '7.0.6', [541], '7.1', '7.0'],
            ['CURLOPT_CONNECT_TO', '7.0.6', [542], '7.1', '7.0'],
            ['CURLOPT_DEFAULT_PROTOCOL', '7.0.6', [543], '7.1', '7.0'],
            ['CURLOPT_DNS_INTERFACE', '7.0.6', [544], '7.1', '7.0'],
            ['CURLOPT_DNS_LOCAL_IP4', '7.0.6', [545], '7.1', '7.0'],
            ['CURLOPT_DNS_LOCAL_IP6', '7.0.6', [546], '7.1', '7.0'],
            ['CURLOPT_EXPECT_100_TIMEOUT_MS', '7.0.6', [547], '7.1', '7.0'],
            ['CURLOPT_HEADEROPT', '7.0.6', [548], '7.1', '7.0'],
            ['CURLOPT_LOGIN_OPTIONS', '7.0.6', [549], '7.1', '7.0'],
            ['CURLOPT_PATH_AS_IS', '7.0.6', [550], '7.1', '7.0'],
            ['CURLOPT_PINNEDPUBLICKEY', '7.0.6', [551], '7.1', '7.0'],
            ['CURLOPT_PIPEWAIT', '7.0.6', [552], '7.1', '7.0'],
            ['CURLOPT_PROXY_SERVICE_NAME', '7.0.6', [553], '7.1', '7.0'],
            ['CURLOPT_PROXYHEADER', '7.0.6', [554], '7.1', '7.0'],
            ['CURLOPT_SASL_IR', '7.0.6', [555], '7.1', '7.0'],
            ['CURLOPT_SERVICE_NAME', '7.0.6', [556], '7.1', '7.0'],
            ['CURLOPT_SSL_ENABLE_ALPN', '7.0.6', [557], '7.1', '7.0'],
            ['CURLOPT_SSL_ENABLE_NPN', '7.0.6', [558], '7.1', '7.0'],
            ['CURLOPT_SSL_FALSESTART', '7.0.6', [559], '7.1', '7.0'],
            ['CURLOPT_SSL_VERIFYSTATUS', '7.0.6', [560], '7.1', '7.0'],
            ['CURLOPT_STREAM_WEIGHT', '7.0.6', [561], '7.1', '7.0'],
            ['CURLOPT_TCP_FASTOPEN', '7.0.6', [562], '7.1', '7.0'],
            ['CURLOPT_TFTP_NO_OPTIONS', '7.0.6', [563], '7.1', '7.0'],
            ['CURLOPT_UNIX_SOCKET_PATH', '7.0.6', [564], '7.1', '7.0'],
            ['CURLOPT_XOAUTH2_BEARER', '7.0.6', [565], '7.1', '7.0'],
            ['CURLPROTO_SMB', '7.0.6', [566], '7.1', '7.0'],
            ['CURLPROTO_SMBS', '7.0.6', [567], '7.1', '7.0'],
            ['CURLPROXY_HTTP_1_0', '7.0.6', [568], '7.1', '7.0'],
            ['CURLSSH_AUTH_AGENT', '7.0.6', [569], '7.1', '7.0'],
            ['CURLSSLOPT_NO_REVOKE', '7.0.6', [570], '7.1', '7.0'],

            ['DNS_CAA', '7.0.15', [1059], '7.1', '7.0'],
            ['DATE_RFC7231', '7.0.18', [884], '7.1', '7.0'],

            ['PHP_FD_SETSIZE', '7.0', [571], '7.1'],
            ['CURLMOPT_PUSHFUNCTION', '7.0', [572], '7.1'],
            ['CURL_PUSH_OK', '7.0', [573], '7.1'],
            ['CURL_PUSH_DENY', '7.0', [574], '7.1'],
            ['FILTER_FLAG_EMAIL_UNICODE', '7.0', [575], '7.1'],
            ['IMAGETYPE_WEBP', '7.0', [576], '7.1'],
            ['JSON_UNESCAPED_LINE_TERMINATORS', '7.0', [577], '7.1'],
            ['LDAP_OPT_X_SASL_NOCANON', '7.0', [578], '7.1'],
            ['LDAP_OPT_X_SASL_USERNAME', '7.0', [579], '7.1'],
            ['LDAP_OPT_X_TLS_CACERTDIR', '7.0', [580], '7.1'],
            ['LDAP_OPT_X_TLS_CACERTFILE', '7.0', [581], '7.1'],
            ['LDAP_OPT_X_TLS_CERTFILE', '7.0', [582], '7.1'],
            ['LDAP_OPT_X_TLS_CIPHER_SUITE', '7.0', [583], '7.1'],
            ['LDAP_OPT_X_TLS_KEYFILE', '7.0', [584], '7.1'],
            ['LDAP_OPT_X_TLS_RANDOM_FILE', '7.0', [585], '7.1'],
            ['LDAP_OPT_X_TLS_CRLCHECK', '7.0', [586], '7.1'],
            ['LDAP_OPT_X_TLS_CRL_NONE', '7.0', [587], '7.1'],
            ['LDAP_OPT_X_TLS_CRL_PEER', '7.0', [588], '7.1'],
            ['LDAP_OPT_X_TLS_CRL_ALL', '7.0', [589], '7.1'],
            ['LDAP_OPT_X_TLS_DHFILE', '7.0', [590], '7.1'],
            ['LDAP_OPT_X_TLS_CRLFILE', '7.0', [591], '7.1'],
            ['LDAP_OPT_X_TLS_PROTOCOL_MIN', '7.0', [592], '7.1'],
            ['LDAP_OPT_X_TLS_PROTOCOL_SSL2', '7.0', [593], '7.1'],
            ['LDAP_OPT_X_TLS_PROTOCOL_SSL3', '7.0', [594], '7.1'],
            ['LDAP_OPT_X_TLS_PROTOCOL_TLS1_0', '7.0', [595], '7.1'],
            ['LDAP_OPT_X_TLS_PROTOCOL_TLS1_1', '7.0', [596], '7.1'],
            ['LDAP_OPT_X_TLS_PROTOCOL_TLS1_2', '7.0', [597], '7.1'],
            ['LDAP_OPT_X_TLS_PACKAGE', '7.0', [598], '7.1'],
            ['LDAP_OPT_X_KEEPALIVE_IDLE', '7.0', [599], '7.1'],
            ['LDAP_OPT_X_KEEPALIVE_PROBES', '7.0', [600], '7.1'],
            ['LDAP_OPT_X_KEEPALIVE_INTERVAL', '7.0', [601], '7.1'],
            ['PGSQL_NOTICE_LAST', '7.0', [602], '7.1'],
            ['PGSQL_NOTICE_ALL', '7.0', [603], '7.1'],
            ['PGSQL_NOTICE_CLEAR', '7.0', [604], '7.1'],
            ['MT_RAND_PHP', '7.0', [605], '7.1'],

            ['SQLITE3_DETERMINISTIC', '7.1.3', [606], '7.2', '7.1'],
            ['PHP_OS_FAMILY', '7.1', [607], '7.2'],
            ['PHP_FLOAT_DIG', '7.1', [608], '7.2'],
            ['PHP_FLOAT_EPSILON', '7.1', [609], '7.2'],
            ['PHP_FLOAT_MIN', '7.1', [610], '7.2'],
            ['PHP_FLOAT_MAX', '7.1', [611], '7.2'],
            ['PASSWORD_ARGON2I', '7.1', [612], '7.2'],
            ['PASSWORD_ARGON2_DEFAULT_MEMORY_COST', '7.1', [613], '7.2'],
            ['PASSWORD_ARGON2_DEFAULT_TIME_COST', '7.1', [614], '7.2'],
            ['PASSWORD_ARGON2_DEFAULT_THREADS', '7.1', [615], '7.2'],
            ['FILEINFO_EXTENSION', '7.1', [616], '7.2'],
            ['IMG_EFFECT_MULTIPLY', '7.1', [617], '7.2'],
            ['IMG_BMP', '7.1', [618], '7.2'],
            ['JSON_INVALID_UTF8_IGNORE', '7.1', [619], '7.2'],
            ['JSON_INVALID_UTF8_SUBSTITUTE', '7.1', [620], '7.2'],
            ['LDAP_EXOP_START_TLS', '7.1', [621], '7.2'],
            ['LDAP_EXOP_MODIFY_PASSWD', '7.1', [622], '7.2'],
            ['LDAP_EXOP_REFRESH', '7.1', [623], '7.2'],
            ['LDAP_EXOP_WHO_AM_I', '7.1', [624], '7.2'],
            ['LDAP_EXOP_TURN', '7.1', [625], '7.2'],
            ['PREG_UNMATCHED_AS_NULL', '7.1', [626], '7.2'],
            ['SODIUM_LIBRARY_VERSION', '7.1', [627], '7.2'],
            ['SODIUM_LIBRARY_MAJOR_VERSION', '7.1', [628], '7.2'],
            ['SODIUM_LIBRARY_MINOR_VERSION', '7.1', [629], '7.2'],
            ['SODIUM_BASE64_VARIANT_ORIGINAL', '7.1', [866], '7.2'],
            ['SODIUM_BASE64_VARIANT_ORIGINAL_NO_PADDING', '7.1', [867], '7.2'],
            ['SODIUM_BASE64_VARIANT_URLSAFE', '7.1', [868], '7.2'],
            ['SODIUM_BASE64_VARIANT_URLSAFE_NO_PADDING', '7.1', [869], '7.2'],
            ['SODIUM_CRYPTO_AEAD_AES256GCM_KEYBYTES', '7.1', [630], '7.2'],
            ['SODIUM_CRYPTO_AEAD_AES256GCM_NSECBYTES', '7.1', [631], '7.2'],
            ['SODIUM_CRYPTO_AEAD_AES256GCM_NPUBBYTES', '7.1', [632], '7.2'],
            ['SODIUM_CRYPTO_AEAD_AES256GCM_ABYTES', '7.1', [633], '7.2'],
            ['SODIUM_CRYPTO_AEAD_CHACHA20POLY1305_KEYBYTES', '7.1', [634], '7.2'],
            ['SODIUM_CRYPTO_AEAD_CHACHA20POLY1305_NSECBYTES', '7.1', [635], '7.2'],
            ['SODIUM_CRYPTO_AEAD_CHACHA20POLY1305_NPUBBYTES', '7.1', [636], '7.2'],
            ['SODIUM_CRYPTO_AEAD_CHACHA20POLY1305_ABYTES', '7.1', [637], '7.2'],
            ['SODIUM_CRYPTO_AEAD_CHACHA20POLY1305_IETF_KEYBYTES', '7.1', [638], '7.2'],
            ['SODIUM_CRYPTO_AEAD_CHACHA20POLY1305_IETF_NSECBYTES', '7.1', [639], '7.2'],
            ['SODIUM_CRYPTO_AEAD_CHACHA20POLY1305_IETF_NPUBBYTES', '7.1', [640], '7.2'],
            ['SODIUM_CRYPTO_AEAD_CHACHA20POLY1305_IETF_ABYTES', '7.1', [641], '7.2'],
            ['SODIUM_CRYPTO_AEAD_XCHACHA20POLY1305_IETF_KEYBYTES', '7.1', [870], '7.2'],
            ['SODIUM_CRYPTO_AEAD_XCHACHA20POLY1305_IETF_NSECBYTES', '7.1', [871], '7.2'],
            ['SODIUM_CRYPTO_AEAD_XCHACHA20POLY1305_IETF_NPUBBYTES', '7.1', [872], '7.2'],
            ['SODIUM_CRYPTO_AEAD_XCHACHA20POLY1305_IETF_ABYTES', '7.1', [873], '7.2'],
            ['SODIUM_CRYPTO_AUTH_BYTES', '7.1', [642], '7.2'],
            ['SODIUM_CRYPTO_AUTH_KEYBYTES', '7.1', [643], '7.2'],
            ['SODIUM_CRYPTO_BOX_SEALBYTES', '7.1', [644], '7.2'],
            ['SODIUM_CRYPTO_BOX_SECRETKEYBYTES', '7.1', [645], '7.2'],
            ['SODIUM_CRYPTO_BOX_PUBLICKEYBYTES', '7.1', [646], '7.2'],
            ['SODIUM_CRYPTO_BOX_KEYPAIRBYTES', '7.1', [647], '7.2'],
            ['SODIUM_CRYPTO_BOX_MACBYTES', '7.1', [648], '7.2'],
            ['SODIUM_CRYPTO_BOX_NONCEBYTES', '7.1', [649], '7.2'],
            ['SODIUM_CRYPTO_BOX_SEEDBYTES', '7.1', [650], '7.2'],
            ['SODIUM_CRYPTO_KDF_BYTES_MIN', '7.1', [651], '7.2'],
            ['SODIUM_CRYPTO_KDF_BYTES_MAX', '7.1', [652], '7.2'],
            ['SODIUM_CRYPTO_KDF_CONTEXTBYTES', '7.1', [653], '7.2'],
            ['SODIUM_CRYPTO_KDF_KEYBYTES', '7.1', [654], '7.2'],
            ['SODIUM_CRYPTO_KX_SEEDBYTES', '7.1', [655], '7.2'],
            ['SODIUM_CRYPTO_KX_SESSIONKEYBYTES', '7.1', [656], '7.2'],
            ['SODIUM_CRYPTO_KX_PUBLICKEYBYTES', '7.1', [657], '7.2'],
            ['SODIUM_CRYPTO_KX_SECRETKEYBYTES', '7.1', [658], '7.2'],
            ['SODIUM_CRYPTO_KX_KEYPAIRBYTES', '7.1', [659], '7.2'],
            ['SODIUM_CRYPTO_GENERICHASH_BYTES', '7.1', [660], '7.2'],
            ['SODIUM_CRYPTO_GENERICHASH_BYTES_MIN', '7.1', [661], '7.2'],
            ['SODIUM_CRYPTO_GENERICHASH_BYTES_MAX', '7.1', [662], '7.2'],
            ['SODIUM_CRYPTO_GENERICHASH_KEYBYTES', '7.1', [663], '7.2'],
            ['SODIUM_CRYPTO_GENERICHASH_KEYBYTES_MIN', '7.1', [664], '7.2'],
            ['SODIUM_CRYPTO_GENERICHASH_KEYBYTES_MAX', '7.1', [665], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_ALG_ARGON2I13', '7.1', [666], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_ALG_ARGON2ID13', '7.1', [865], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_ALG_DEFAULT', '7.1', [667], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_SALTBYTES', '7.1', [668], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_STRPREFIX', '7.1', [669], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_OPSLIMIT_INTERACTIVE', '7.1', [670], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_MEMLIMIT_INTERACTIVE', '7.1', [671], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_OPSLIMIT_MODERATE', '7.1', [672], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_MEMLIMIT_MODERATE', '7.1', [673], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_OPSLIMIT_SENSITIVE', '7.1', [674], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_MEMLIMIT_SENSITIVE', '7.1', [675], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_SCRYPTSALSA208SHA256_SALTBYTES', '7.1', [676], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_SCRYPTSALSA208SHA256_STRPREFIX', '7.1', [677], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_SCRYPTSALSA208SHA256_OPSLIMIT_INTERACTIVE', '7.1', [678], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_SCRYPTSALSA208SHA256_MEMLIMIT_INTERACTIVE', '7.1', [679], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_SCRYPTSALSA208SHA256_OPSLIMIT_SENSITIVE', '7.1', [680], '7.2'],
            ['SODIUM_CRYPTO_PWHASH_SCRYPTSALSA208SHA256_MEMLIMIT_SENSITIVE', '7.1', [681], '7.2'],
            ['SODIUM_CRYPTO_SCALARMULT_BYTES', '7.1', [682], '7.2'],
            ['SODIUM_CRYPTO_SCALARMULT_SCALARBYTES', '7.1', [683], '7.2'],
            ['SODIUM_CRYPTO_SECRETSTREAM_XCHACHA20POLY1305_ABYTES', '7.1', [874], '7.2'],
            ['SODIUM_CRYPTO_SECRETSTREAM_XCHACHA20POLY1305_HEADERBYTES', '7.1', [875], '7.2'],
            ['SODIUM_CRYPTO_SECRETSTREAM_XCHACHA20POLY1305_KEYBYTES', '7.1', [876], '7.2'],
            ['SODIUM_CRYPTO_SECRETSTREAM_XCHACHA20POLY1305_MESSAGEBYTES_MAX', '7.1', [877], '7.2'],
            ['SODIUM_CRYPTO_SECRETSTREAM_XCHACHA20POLY1305_TAG_MESSAGE', '7.1', [878], '7.2'],
            ['SODIUM_CRYPTO_SECRETSTREAM_XCHACHA20POLY1305_TAG_PUSH', '7.1', [879], '7.2'],
            ['SODIUM_CRYPTO_SECRETSTREAM_XCHACHA20POLY1305_TAG_REKEY', '7.1', [880], '7.2'],
            ['SODIUM_CRYPTO_SECRETSTREAM_XCHACHA20POLY1305_TAG_FINAL', '7.1', [881], '7.2'],
            ['SODIUM_CRYPTO_SHORTHASH_BYTES', '7.1', [684], '7.2'],
            ['SODIUM_CRYPTO_SHORTHASH_KEYBYTES', '7.1', [685], '7.2'],
            ['SODIUM_CRYPTO_SECRETBOX_KEYBYTES', '7.1', [686], '7.2'],
            ['SODIUM_CRYPTO_SECRETBOX_MACBYTES', '7.1', [687], '7.2'],
            ['SODIUM_CRYPTO_SECRETBOX_NONCEBYTES', '7.1', [688], '7.2'],
            ['SODIUM_CRYPTO_SIGN_BYTES', '7.1', [689], '7.2'],
            ['SODIUM_CRYPTO_SIGN_SEEDBYTES', '7.1', [690], '7.2'],
            ['SODIUM_CRYPTO_SIGN_PUBLICKEYBYTES', '7.1', [691], '7.2'],
            ['SODIUM_CRYPTO_SIGN_SECRETKEYBYTES', '7.1', [692], '7.2'],
            ['SODIUM_CRYPTO_SIGN_KEYPAIRBYTES', '7.1', [693], '7.2'],
            ['SODIUM_CRYPTO_STREAM_NONCEBYTES', '7.1', [694], '7.2'],
            ['SODIUM_CRYPTO_STREAM_KEYBYTES', '7.1', [695], '7.2'],

            ['CURLOPT_REQUEST_TARGET', '7.2', [699], '7.3'],
            ['FILTER_SANITIZE_ADD_SLASHES', '7.2', [700], '7.3'],
            ['JSON_THROW_ON_ERROR', '7.2', [701], '7.3'],
            ['MB_CASE_FOLD', '7.2', [702], '7.3'],
            ['MB_CASE_UPPER_SIMPLE', '7.2', [703], '7.3'],
            ['MB_CASE_LOWER_SIMPLE', '7.2', [704], '7.3'],
            ['MB_CASE_TITLE_SIMPLE', '7.2', [705], '7.3'],
            ['MB_CASE_FOLD_SIMPLE', '7.2', [706], '7.3'],
            ['PGSQL_DIAG_SCHEMA_NAME', '7.2', [707], '7.3'],
            ['PGSQL_DIAG_TABLE_NAME', '7.2', [708], '7.3'],
            ['PGSQL_DIAG_COLUMN_NAME', '7.2', [709], '7.3'],
            ['PGSQL_DIAG_DATATYPE_NAME', '7.2', [710], '7.3'],
            ['PGSQL_DIAG_CONSTRAINT_NAME', '7.2', [711], '7.3'],
            ['PGSQL_DIAG_SEVERITY_NONLOCALIZED', '7.2', [712], '7.3'],
            ['PASSWORD_ARGON2ID', '7.2', [713], '7.3'],
            ['CURLAUTH_GSSAPI', '7.2', [714], '7.3'],
            ['CURLE_WEIRD_SERVER_REPLY', '7.2', [715], '7.3'],
            ['CURLINFO_CONTENT_LENGTH_DOWNLOAD_T', '7.2', [716], '7.3'],
            ['CURLINFO_CONTENT_LENGTH_UPLOAD_T', '7.2', [717], '7.3'],
            ['CURLINFO_HTTP_VERSION', '7.2', [718], '7.3'],
            ['CURLINFO_PROTOCOL', '7.2', [719], '7.3'],
            ['CURLINFO_PROXY_SSL_VERIFYRESULT', '7.2', [720], '7.3'],
            ['CURLINFO_SCHEME', '7.2', [721], '7.3'],
            ['CURLINFO_SIZE_DOWNLOAD_T', '7.2', [722], '7.3'],
            ['CURLINFO_SIZE_UPLOAD_T', '7.2', [723], '7.3'],
            ['CURLINFO_SPEED_DOWNLOAD_T', '7.2', [724], '7.3'],
            ['CURLINFO_SPEED_UPLOAD_T', '7.2', [725], '7.3'],
            ['CURL_MAX_READ_SIZE', '7.2', [726], '7.3'],
            ['CURLOPT_ABSTRACT_UNIX_SOCKET', '7.2', [727], '7.3'],
            ['CURLOPT_KEEP_SENDING_ON_ERROR', '7.2', [728], '7.3'],
            ['CURLOPT_PRE_PROXY', '7.2', [729], '7.3'],
            ['CURLOPT_PROXY_CAINFO', '7.2', [730], '7.3'],
            ['CURLOPT_PROXY_CAPATH', '7.2', [731], '7.3'],
            ['CURLOPT_PROXY_CRLFILE', '7.2', [732], '7.3'],
            ['CURLOPT_PROXY_KEYPASSWD', '7.2', [733], '7.3'],
            ['CURLOPT_PROXY_PINNEDPUBLICKEY', '7.2', [734], '7.3'],
            ['CURLOPT_PROXY_SSLCERT', '7.2', [735], '7.3'],
            ['CURLOPT_PROXY_SSLCERTTYPE', '7.2', [736], '7.3'],
            ['CURLOPT_PROXY_SSL_CIPHER_LIST', '7.2', [737], '7.3'],
            ['CURLOPT_PROXY_SSLKEY', '7.2', [738], '7.3'],
            ['CURLOPT_PROXY_SSLKEYTYPE', '7.2', [739], '7.3'],
            ['CURLOPT_PROXY_SSL_OPTIONS', '7.2', [740], '7.3'],
            ['CURLOPT_PROXY_SSL_VERIFYHOST', '7.2', [741], '7.3'],
            ['CURLOPT_PROXY_SSL_VERIFYPEER', '7.2', [742], '7.3'],
            ['CURLOPT_PROXY_SSLVERSION', '7.2', [743], '7.3'],
            ['CURLOPT_PROXY_TLSAUTH_PASSWORD', '7.2', [744], '7.3'],
            ['CURLOPT_PROXY_TLSAUTH_TYPE', '7.2', [745], '7.3'],
            ['CURLOPT_PROXY_TLSAUTH_USERNAME', '7.2', [746], '7.3'],
            ['CURLOPT_SOCKS5_AUTH', '7.2', [747], '7.3'],
            ['CURLOPT_SUPPRESS_CONNECT_HEADERS', '7.2', [748], '7.3'],
            ['CURLPROXY_HTTPS', '7.2', [749], '7.3'],
            ['CURL_SSLVERSION_MAX_DEFAULT', '7.2', [750], '7.3'],
            ['CURL_SSLVERSION_MAX_NONE', '7.2', [751], '7.3'],
            ['CURL_SSLVERSION_MAX_TLSv1_0', '7.2', [752], '7.3'],
            ['CURL_SSLVERSION_MAX_TLSv1_1', '7.2', [753], '7.3'],
            ['CURL_SSLVERSION_MAX_TLSv1_2', '7.2', [754], '7.3'],
            ['CURL_SSLVERSION_MAX_TLSv1_3', '7.2', [755], '7.3'],
            ['CURL_SSLVERSION_TLSv1_3', '7.2', [756], '7.3'],
            ['CURL_VERSION_HTTPS_PROXY', '7.2', [757], '7.3'],
            ['CURLAUTH_BEARER', '7.2', [758], '7.3'],
            ['CURLINFO_APPCONNECT_TIME_T', '7.2', [759], '7.3'],
            ['CURLINFO_CONNECT_TIME_T', '7.2', [760], '7.3'],
            ['CURLINFO_FILETIME_T', '7.2', [761], '7.3'],
            ['CURLINFO_NAMELOOKUP_TIME_T', '7.2', [762], '7.3'],
            ['CURLINFO_PRETRANSFER_TIME_T', '7.2', [763], '7.3'],
            ['CURLINFO_REDIRECT_TIME_T', '7.2', [764], '7.3'],
            ['CURLINFO_STARTTRANSFER_TIME_T', '7.2', [765], '7.3'],
            ['CURLINFO_TOTAL_TIME_T', '7.2', [766], '7.3'],
            ['CURL_LOCK_DATA_CONNECT', '7.2', [767], '7.3'],
            ['CURL_LOCK_DATA_PSL', '7.2', [768], '7.3'],
            ['CURLOPT_DISALLOW_USERNAME_IN_URL', '7.2', [769], '7.3'],
            ['CURLOPT_DNS_SHUFFLE_ADDRESSES', '7.2', [770], '7.3'],
            ['CURLOPT_HAPPY_EYEBALLS_TIMEOUT_MS', '7.2', [771], '7.3'],
            ['CURLOPT_HAPROXYPROTOCOL', '7.2', [772], '7.3'],
            ['CURLOPT_PROXY_TLS13_CIPHERS', '7.2', [773], '7.3'],
            ['CURLOPT_SSH_COMPRESSION', '7.2', [774], '7.3'],
            ['CURLOPT_TIMEVALUE_LARGE', '7.2', [775], '7.3'],
            ['CURLOPT_TLS13_CIPHERS', '7.2', [776], '7.3'],
            ['CURLSSH_AUTH_GSSAPI', '7.2', [777], '7.3'],
            ['CURL_VERSION_ASYNCHDNS', '7.2', [778], '7.3'],
            ['CURL_VERSION_BROTLI', '7.2', [779], '7.3'],
            ['CURL_VERSION_CONV', '7.2', [780], '7.3'],
            ['CURL_VERSION_DEBUG', '7.2', [781], '7.3'],
            ['CURL_VERSION_GSSAPI', '7.2', [782], '7.3'],
            ['CURL_VERSION_GSSNEGOTIATE', '7.2', [783], '7.3'],
            ['CURL_VERSION_IDN', '7.2', [784], '7.3'],
            ['CURL_VERSION_LARGEFILE', '7.2', [785], '7.3'],
            ['CURL_VERSION_MULTI_SSL', '7.2', [786], '7.3'],
            ['CURL_VERSION_NTLM', '7.2', [787], '7.3'],
            ['CURL_VERSION_NTLM_WB', '7.2', [788], '7.3'],
            ['CURL_VERSION_SPNEGO', '7.2', [789], '7.3'],
            ['CURL_VERSION_SSPI', '7.2', [790], '7.3'],
            ['CURL_VERSION_TLSAUTH_SRP', '7.2', [791], '7.3'],
            ['STREAM_CRYPTO_PROTO_SSLv3', '7.2', [792], '7.3'],
            ['STREAM_CRYPTO_PROTO_TLSv1_0', '7.2', [793], '7.3'],
            ['STREAM_CRYPTO_PROTO_TLSv1_1', '7.2', [794], '7.3'],
            ['STREAM_CRYPTO_PROTO_TLSv1_2', '7.2', [795], '7.3'],
            ['LDAP_CONTROL_MANAGEDSAIT', '7.2', [796], '7.3'],
            ['LDAP_CONTROL_PROXY_AUTHZ', '7.2', [797], '7.3'],
            ['LDAP_CONTROL_SUBENTRIES', '7.2', [798], '7.3'],
            ['LDAP_CONTROL_VALUESRETURNFILTER', '7.2', [799], '7.3'],
            ['LDAP_CONTROL_ASSERT', '7.2', [800], '7.3'],
            ['LDAP_CONTROL_PRE_READ', '7.2', [801], '7.3'],
            ['LDAP_CONTROL_POST_READ', '7.2', [802], '7.3'],
            ['LDAP_CONTROL_SORTREQUEST', '7.2', [803], '7.3'],
            ['LDAP_CONTROL_SORTRESPONSE', '7.2', [804], '7.3'],
            ['LDAP_CONTROL_PAGEDRESULTS', '7.2', [805], '7.3'],
            ['LDAP_CONTROL_AUTHZID_REQUEST', '7.2', [806], '7.3'],
            ['LDAP_CONTROL_AUTHZID_RESPONSE', '7.2', [807], '7.3'],
            ['LDAP_CONTROL_SYNC', '7.2', [808], '7.3'],
            ['LDAP_CONTROL_SYNC_STATE', '7.2', [809], '7.3'],
            ['LDAP_CONTROL_SYNC_DONE', '7.2', [810], '7.3'],
            ['LDAP_CONTROL_DONTUSECOPY', '7.2', [811], '7.3'],
            ['LDAP_CONTROL_PASSWORDPOLICYREQUEST', '7.2', [812], '7.3'],
            ['LDAP_CONTROL_PASSWORDPOLICYRESPONSE', '7.2', [813], '7.3'],
            ['LDAP_CONTROL_X_INCREMENTAL_VALUES', '7.2', [814], '7.3'],
            ['LDAP_CONTROL_X_DOMAIN_SCOPE', '7.2', [815], '7.3'],
            ['LDAP_CONTROL_X_PERMISSIVE_MODIFY', '7.2', [816], '7.3'],
            ['LDAP_CONTROL_X_SEARCH_OPTIONS', '7.2', [817], '7.3'],
            ['LDAP_CONTROL_X_TREE_DELETE', '7.2', [818], '7.3'],
            ['LDAP_CONTROL_X_EXTENDED_DN', '7.2', [819], '7.3'],
            ['LDAP_CONTROL_VLVREQUEST', '7.2', [820], '7.3'],
            ['LDAP_CONTROL_VLVRESPONSE', '7.2', [821], '7.3'],

            ['CURL_VERSION_ALTSVC', '7.3.5', [860], '7.4', '7.3'],
            ['CURL_VERSION_CURLDEBUG', '7.3.5', [861], '7.4', '7.3'],
            ['CURLOPT_HTTP09_ALLOWED', '7.3.14', [1056], '7.4', '7.3'],

            ['IMG_FILTER_SCATTER', '7.3', [863], '7.4'],
            ['MB_ONIGURUMA_VERSION', '7.3', [824], '7.4'],
            ['SO_LABEL', '7.3', [825], '7.4'],
            ['SO_PEERLABEL', '7.3', [826], '7.4'],
            ['SO_LISTENQLIMIT', '7.3', [827], '7.4'],
            ['SO_LISTENQLEN', '7.3', [828], '7.4'],
            ['SO_USER_COOKIE', '7.3', [829], '7.4'],
            ['PASSWORD_ARGON2_PROVIDER', '7.3', [864], '7.4'],
            ['PHP_WINDOWS_EVENT_CTRL_C', '7.3', [830], '7.4'],
            ['PHP_WINDOWS_EVENT_CTRL_BREAK', '7.3', [831], '7.4'],
            ['T_BAD_CHARACTER', '7.3', [862], '7.4'],
            ['T_COALESCE_EQUAL', '7.3', [1048], '7.4'],
            ['T_FN', '7.3', [1049], '7.4'],
            ['TIDY_TAG_ARTICLE', '7.3', [832], '7.4'],
            ['TIDY_TAG_ASIDE', '7.3', [833], '7.4'],
            ['TIDY_TAG_AUDIO', '7.3', [834], '7.4'],
            ['TIDY_TAG_BDI', '7.3', [835], '7.4'],
            ['TIDY_TAG_CANVAS', '7.3', [836], '7.4'],
            ['TIDY_TAG_COMMAND', '7.3', [837], '7.4'],
            ['TIDY_TAG_DATALIST', '7.3', [838], '7.4'],
            ['TIDY_TAG_DETAILS', '7.3', [839], '7.4'],
            ['TIDY_TAG_DIALOG', '7.3', [840], '7.4'],
            ['TIDY_TAG_FIGCAPTION', '7.3', [841], '7.4'],
            ['TIDY_TAG_FIGURE', '7.3', [842], '7.4'],
            ['TIDY_TAG_FOOTER', '7.3', [843], '7.4'],
            ['TIDY_TAG_HEADER', '7.3', [844], '7.4'],
            ['TIDY_TAG_HGROUP', '7.3', [845], '7.4'],
            ['TIDY_TAG_MAIN', '7.3', [846], '7.4'],
            ['TIDY_TAG_MARK', '7.3', [847], '7.4'],
            ['TIDY_TAG_MENUITEM', '7.3', [848], '7.4'],
            ['TIDY_TAG_METER', '7.3', [849], '7.4'],
            ['TIDY_TAG_NAV', '7.3', [850], '7.4'],
            ['TIDY_TAG_OUTPUT', '7.3', [851], '7.4'],
            ['TIDY_TAG_PROGRESS', '7.3', [852], '7.4'],
            ['TIDY_TAG_SECTION', '7.3', [853], '7.4'],
            ['TIDY_TAG_SOURCE', '7.3', [854], '7.4'],
            ['TIDY_TAG_SUMMARY', '7.3', [855], '7.4'],
            ['TIDY_TAG_TEMPLATE', '7.3', [856], '7.4'],
            ['TIDY_TAG_TIME', '7.3', [857], '7.4'],
            ['TIDY_TAG_TRACK', '7.3', [858], '7.4'],
            ['TIDY_TAG_VIDEO', '7.3', [859], '7.4'],

            ['FILTER_VALIDATE_BOOL', '7.4', [1407], '8.0'],
            ['LIBENCHANT_VERSION', '7.4', [1408], '8.0'],
            ['OPENSSL_ENCODING_DER', '7.4', [1396], '8.0'],
            ['OPENSSL_ENCODING_SMIME', '7.4', [1397], '8.0'],
            ['OPENSSL_ENCODING_PEM', '7.4', [1398], '8.0'],
            ['OPENSSL_CMS_DETACHED', '7.4', [1399], '8.0'],
            ['OPENSSL_CMS_TEXT', '7.4', [1400], '8.0'],
            ['OPENSSL_CMS_NOINTERN', '7.4', [1401], '8.0'],
            ['OPENSSL_CMS_NOVERIFY', '7.4', [1402], '8.0'],
            ['OPENSSL_CMS_NOCERTS', '7.4', [1403], '8.0'],
            ['OPENSSL_CMS_NOATTR', '7.4', [1404], '8.0'],
            ['OPENSSL_CMS_BINARY', '7.4', [1405], '8.0'],
            ['OPENSSL_CMS_NOSIGS', '7.4', [1406], '8.0'],
            ['T_MATCH', '7.4', [1413], '8.0'],
            ['T_NAME_FULLY_QUALIFIED', '7.4', [1410], '8.0'],
            ['T_NAME_QUALIFIED', '7.4', [1411], '8.0'],
            ['T_NAME_RELATIVE', '7.4', [1412], '8.0'],
            ['T_NULLSAFE_OBJECT_OPERATOR', '7.4', [1409], '8.0'],

            ['CURLOPT_DOH_URL', '8.0', [1414], '8.1'],
            ['CURLOPT_ISSUERCERT_BLOB', '8.0', [1415], '8.1'],
            ['CURLOPT_PROXY_ISSUERCERT', '8.0', [1416], '8.1'],
            ['CURLOPT_PROXY_ISSUERCERT_BLOB', '8.0', [1417], '8.1'],
            ['CURLOPT_PROXY_SSLCERT_BLOB', '8.0', [1418], '8.1'],
            ['CURLOPT_PROXY_SSLKEY_BLOB', '8.0', [1419], '8.1'],
            ['CURLOPT_SSLCERT_BLOB', '8.0', [1420], '8.1'],
            ['CURLOPT_SSLKEY_BLOB', '8.0', [1421], '8.1'],
            ['IMG_AVIF', '8.0', [1422], '8.1'],
            ['IMG_WEBP_LOSSLESS', '8.0', [1423], '8.1'],
            ['MYSQLI_REFRESH_REPLICA', '8.0', [1424], '8.1'],
            ['POSIX_RLIMIT_KQUEUES', '8.0', [1425], '8.1'],
            ['POSIX_RLIMIT_NPTS', '8.0', [1426], '8.1'],
            ['SO_ACCEPTFILTER', '8.0', [1427], '8.1'],
            ['SO_DONTTRUNC', '8.0', [1428], '8.1'],
            ['SO_WANTMORE', '8.0', [1429], '8.1'],
            ['SO_MARK', '8.0', [1430], '8.1'],
            ['TCP_DEFER_ACCEPT', '8.0', [1431], '8.1'],
            ['T_READONLY', '8.0', [1432], '8.1'],
            ['SODIUM_CRYPTO_STREAM_XCHACHA20_NONCEBYTES', '8.0', [1433], '8.1'],
            ['SODIUM_CRYPTO_STREAM_XCHACHA20_KEYBYTES', '8.0', [1434], '8.1'],
            ['SODIUM_CRYPTO_SCALARMULT_RISTRETTO255_BYTES', '8.0', [1435], '8.1'],
            ['SODIUM_CRYPTO_SCALARMULT_RISTRETTO255_SCALARBYTES', '8.0', [1436], '8.1'],
            ['SODIUM_CRYPTO_CORE_RISTRETTO255_BYTES', '8.0', [1437], '8.1'],
            ['SODIUM_CRYPTO_CORE_RISTRETTO255_HASHBYTES', '8.0', [1438], '8.1'],
            ['SODIUM_CRYPTO_CORE_RISTRETTO255_SCALARBYTES', '8.0', [1439], '8.1'],
            ['SODIUM_CRYPTO_CORE_RISTRETTO255_NONREDUCEDSCALARBYTES', '8.0', [1440], '8.1'],
        ];
    }


    /**
     * Test constants that shouldn't be flagged by this sniff.
     *
     * @dataProvider dataNoFalsePositives
     *
     * @param int $line The line number.
     *
     * @return void
     */
    public function testNoFalsePositives($line)
    {
        $file = $this->sniffFile(__FILE__, '4.4'); // Low version below the first addition.
        $this->assertNoViolation($file, $line);
    }

    /**
     * Data provider.
     *
     * @see testNoFalsePositives()
     *
     * @return array
     */
    public function dataNoFalsePositives()
    {
        return [
            [4],
            [5],
        ];
    }


    /**
     * Verify no notices are thrown at all.
     *
     * @return void
     */
    public function testNoViolationsInFileOnValidVersion()
    {
        $file = $this->sniffFile(__FILE__, '99.0'); // High version beyond newest addition.
        $this->assertNoViolation($file);
    }
}
