<?php
/**
 * PHPCompatibility, an external standard for PHP_CodeSniffer.
 *
 * @package   PHPCompatibility
 * @copyright 2012-2020 PHPCompatibility Contributors
 * @license   https://opensource.org/licenses/LGPL-3.0 LGPL3
 * @link      https://github.com/PHPCompatibility/PHPCompatibility
 */

namespace PHPCompatibility\Tests\FunctionUse;

use PHPCompatibility\Tests\BaseSniffTest;

/**
 * Test the NewFunctions sniff.
 *
 * @group newFunctions
 * @group functionUse
 *
 * @covers \PHPCompatibility\Sniffs\FunctionUse\NewFunctionsSniff
 *
 * @since 5.5
 */
class NewFunctionsUnitTest extends BaseSniffTest
{

    /**
     * testNewFunction
     *
     * @dataProvider dataNewFunction
     *
     * @param string $functionName      Name of the function.
     * @param string $lastVersionBefore The PHP version just *before* the function was introduced.
     * @param array  $lines             The line numbers in the test file which apply to this function.
     * @param string $okVersion         A PHP version in which the function was valid.
     * @param string $testVersion       Optional. A PHP version in which to test for the error if different
     *                                  from the $lastVersionBefore.
     *
     * @return void
     */
    public function testNewFunction($functionName, $lastVersionBefore, $lines, $okVersion, $testVersion = null)
    {
        $errorVersion = (isset($testVersion)) ? $testVersion : $lastVersionBefore;
        $file         = $this->sniffFile(__FILE__, $errorVersion);
        $error        = "The function {$functionName}() is not present in PHP version {$lastVersionBefore} or earlier";
        foreach ($lines as $line) {
            $this->assertError($file, $line, $error);
        }

        $file = $this->sniffFile(__FILE__, $okVersion);
        foreach ($lines as $line) {
            $this->assertNoViolation($file, $line);
        }
    }

    /**
     * Data provider.
     *
     * @see testNewFunction()
     *
     * @return array
     */
    public function dataNewFunction()
    {
        return [
            ['class_implements', '4.4', [700], '5.0'],
            ['class_parents', '4.4', [701], '5.0'],
            ['spl_classes', '4.4', [702], '5.0'],
            ['ob_tidyhandler', '4.4', [713], '5.0'],
            ['tidy_access_count', '4.4', [714], '5.0'],
            ['tidy_config_count', '4.4', [715], '5.0'],
            ['tidy_error_count', '4.4', [716], '5.0'],
            ['tidy_get_output', '4.4', [717], '5.0'],
            ['tidy_warning_count', '4.4', [718], '5.0'],
            ['is_soap_fault', '4.4', [739], '5.0'],
            ['use_soap_error_handler', '4.4', [740], '5.0'],
            ['dom_import_simplexml', '4.4', [741], '5.0'],
            ['simplexml_import_dom', '4.4', [747], '5.0'],
            ['simplexml_load_file', '4.4', [748], '5.0'],
            ['simplexml_load_string', '4.4', [749], '5.0'],
            ['fbsql_set_password', '4.4', [789], '5.0'],
            ['sqlite_array_query', '4.4', [790], '5.0'],
            ['sqlite_busy_timeout', '4.4', [791], '5.0'],
            ['sqlite_changes', '4.4', [792], '5.0'],
            ['sqlite_close', '4.4', [793], '5.0'],
            ['sqlite_column', '4.4', [794], '5.0'],
            ['sqlite_create_aggregate', '4.4', [795], '5.0'],
            ['sqlite_create_function', '4.4', [796], '5.0'],
            ['sqlite_current', '4.4', [797], '5.0'],
            ['sqlite_error_string', '4.4', [798], '5.0'],
            ['sqlite_escape_string', '4.4', [799], '5.0'],
            ['sqlite_exec', '4.4', [800], '5.0'],
            ['sqlite_factory', '4.4', [801], '5.0'],
            ['sqlite_fetch_all', '4.4', [802], '5.0'],
            ['sqlite_fetch_array', '4.4', [803], '5.0'],
            ['sqlite_fetch_column_types', '4.4', [804], '5.0'],
            ['sqlite_fetch_object', '4.4', [805], '5.0'],
            ['sqlite_fetch_single', '4.4', [806], '5.0'],
            ['sqlite_fetch_string', '4.4', [807], '5.0'],
            ['sqlite_field_name', '4.4', [808], '5.0'],
            ['sqlite_has_more', '4.4', [809], '5.0'],
            ['sqlite_has_prev', '4.4', [810], '5.0'],
            ['sqlite_key', '4.4', [811], '5.0'],
            ['sqlite_last_error', '4.4', [812], '5.0'],
            ['sqlite_last_insert_rowid', '4.4', [813], '5.0'],
            ['sqlite_libencoding', '4.4', [814], '5.0'],
            ['sqlite_libversion', '4.4', [815], '5.0'],
            ['sqlite_next', '4.4', [816], '5.0'],
            ['sqlite_num_fields', '4.4', [817], '5.0'],
            ['sqlite_num_rows', '4.4', [818], '5.0'],
            ['sqlite_open', '4.4', [819], '5.0'],
            ['sqlite_popen', '4.4', [820], '5.0'],
            ['sqlite_prev', '4.4', [821], '5.0'],
            ['sqlite_query', '4.4', [822], '5.0'],
            ['sqlite_rewind', '4.4', [823], '5.0'],
            ['sqlite_seek', '4.4', [824], '5.0'],
            ['sqlite_single_query', '4.4', [825], '5.0'],
            ['sqlite_udf_decode_binary', '4.4', [826], '5.0'],
            ['sqlite_udf_encode_binary', '4.4', [827], '5.0'],
            ['sqlite_unbuffered_query', '4.4', [828], '5.0'],
            ['sqlite_valid', '4.4', [829], '5.0'],
            ['mysqli_affected_rows', '4.4', [831], '5.0'],
            ['mysqli_get_client_info', '4.4', [832], '5.0'],
            ['mysqli_get_client_version', '4.4', [833], '5.0'],
            ['mysqli_connect_errno', '4.4', [834], '5.0'],
            ['mysqli_connect_error', '4.4', [835], '5.0'],
            ['mysqli_errno', '4.4', [836], '5.0'],
            ['mysqli_error', '4.4', [837], '5.0'],
            ['mysqli_field_count', '4.4', [838], '5.0'],
            ['mysqli_get_host_info', '4.4', [839], '5.0'],
            ['mysqli_get_proto_info', '4.4', [840], '5.0'],
            ['mysqli_get_server_info', '4.4', [841], '5.0'],
            ['mysqli_get_server_version', '4.4', [842], '5.0'],
            ['mysqli_info', '4.4', [843], '5.0'],
            ['mysqli_insert_id', '4.4', [844], '5.0'],
            ['mysqli_sqlstate', '4.4', [845], '5.0'],
            ['mysqli_warning_count', '4.4', [846], '5.0'],
            ['mysqli_autocommit', '4.4', [847], '5.0'],
            ['mysqli_change_user', '4.4', [848], '5.0'],
            ['mysqli_character_set_name', '4.4', [849], '5.0'],
            ['mysqli_close', '4.4', [850], '5.0'],
            ['mysqli_commit', '4.4', [851], '5.0'],
            ['mysqli_connect', '4.4', [852], '5.0'],
            ['mysqli_debug', '4.4', [853], '5.0'],
            ['mysqli_dump_debug_info', '4.4', [854], '5.0'],
            ['mysqli_init', '4.4', [859], '5.0'],
            ['mysqli_kill', '4.4', [860], '5.0'],
            ['mysqli_more_results', '4.4', [861], '5.0'],
            ['mysqli_multi_query', '4.4', [862], '5.0'],
            ['mysqli_next_result', '4.4', [863], '5.0'],
            ['mysqli_options', '4.4', [864], '5.0'],
            ['mysqli_ping', '4.4', [865], '5.0'],
            ['mysqli_prepare', '4.4', [866], '5.0'],
            ['mysqli_query', '4.4', [867], '5.0'],
            ['mysqli_real_connect', '4.4', [868], '5.0'],
            ['mysqli_real_escape_string', '4.4', [869], '5.0'],
            ['mysqli_real_query', '4.4', [870], '5.0'],
            ['mysqli_rollback', '4.4', [872], '5.0'],
            ['mysqli_select_db', '4.4', [873], '5.0'],
            ['mysqli_set_local_infile_default', '4.4', [875], '5.0'],
            ['mysqli_set_local_infile_handler', '4.4', [876], '5.0'],
            ['mysqli_ssl_set', '4.4', [877], '5.0'],
            ['mysqli_stat', '4.4', [878], '5.0'],
            ['mysqli_stmt_init', '4.4', [879], '5.0'],
            ['mysqli_store_result', '4.4', [880], '5.0'],
            ['mysqli_thread_id', '4.4', [881], '5.0'],
            ['mysqli_thread_safe', '4.4', [882], '5.0'],
            ['mysqli_use_result', '4.4', [883], '5.0'],
            ['mysqli_stmt_affected_rows', '4.4', [884], '5.0'],
            ['mysqli_stmt_errno', '4.4', [885], '5.0'],
            ['mysqli_stmt_error', '4.4', [886], '5.0'],
            ['mysqli_stmt_field_count', '4.4', [887], '5.0'],
            ['mysqli_stmt_insert_id', '4.4', [888], '5.0'],
            ['mysqli_stmt_num_rows', '4.4', [889], '5.0'],
            ['mysqli_stmt_param_count', '4.4', [890], '5.0'],
            ['mysqli_stmt_sqlstate', '4.4', [891], '5.0'],
            ['mysqli_stmt_attr_get', '4.4', [892], '5.0'],
            ['mysqli_stmt_attr_set', '4.4', [893], '5.0'],
            ['mysqli_stmt_bind_param', '4.4', [894], '5.0'],
            ['mysqli_stmt_bind_result', '4.4', [895], '5.0'],
            ['mysqli_stmt_close', '4.4', [896], '5.0'],
            ['mysqli_stmt_data_seek', '4.4', [897], '5.0'],
            ['mysqli_stmt_execute', '4.4', [898], '5.0'],
            ['mysqli_stmt_fetch', '4.4', [899], '5.0'],
            ['mysqli_stmt_free_result', '4.4', [900], '5.0'],
            ['mysqli_stmt_prepare', '4.4', [905], '5.0'],
            ['mysqli_stmt_reset', '4.4', [906], '5.0'],
            ['mysqli_stmt_result_metadata', '4.4', [907], '5.0'],
            ['mysqli_stmt_send_long_data', '4.4', [908], '5.0'],
            ['mysqli_stmt_store_result', '4.4', [909], '5.0'],
            ['mysqli_field_tell', '4.4', [910], '5.0'],
            ['mysqli_num_fields', '4.4', [911], '5.0'],
            ['mysqli_fetch_lengths', '4.4', [912], '5.0'],
            ['mysqli_num_rows', '4.4', [913], '5.0'],
            ['mysqli_data_seek', '4.4', [914], '5.0'],
            ['mysqli_fetch_array', '4.4', [915], '5.0'],
            ['mysqli_fetch_assoc', '4.4', [916], '5.0'],
            ['mysqli_fetch_field_direct', '4.4', [917], '5.0'],
            ['mysqli_fetch_field', '4.4', [918], '5.0'],
            ['mysqli_fetch_fields', '4.4', [919], '5.0'],
            ['mysqli_fetch_object', '4.4', [920], '5.0'],
            ['mysqli_fetch_row', '4.4', [921], '5.0'],
            ['mysqli_field_seek', '4.4', [922], '5.0'],
            ['mysqli_free_result', '4.4', [923], '5.0'],
            ['mysqli_bind_param', '4.4', [926], '5.0'],
            ['mysqli_bind_result', '4.4', [927], '5.0'],
            ['mysqli_client_encoding', '4.4', [928], '5.0'],
            ['mysqli_disable_rpl_parse', '4.4', [929], '5.0'],
            ['mysqli_enable_reads_from_master', '4.4', [930], '5.0'],
            ['mysqli_enable_rpl_parse', '4.4', [931], '5.0'],
            ['mysqli_escape_string', '4.4', [932], '5.0'],
            ['mysqli_execute', '4.4', [933], '5.0'],
            ['mysqli_fetch', '4.4', [934], '5.0'],
            ['mysqli_get_metadata', '4.4', [935], '5.0'],
            ['mysqli_master_query', '4.4', [936], '5.0'],
            ['mysqli_param_count', '4.4', [937], '5.0'],
            ['mysqli_report', '4.4', [938], '5.0'],
            ['mysqli_rpl_parse_enabled', '4.4', [939], '5.0'],
            ['mysqli_rpl_probe', '4.4', [940], '5.0'],
            ['mysqli_send_long_data', '4.4', [941], '5.0'],
            ['mysqli_set_opt', '4.4', [942], '5.0'],
            ['mysqli_slave_query', '4.4', [943], '5.0'],
            ['mysqli_disable_reads_from_master', '4.4', [944], '5.0'],
            ['mysqli_embedded_connect', '4.4', [945], '5.0'],
            ['mysqli_resource', '4.4', [946], '5.0'],
            ['mysqli_server_end', '4.4', [948], '5.0'],
            ['mysqli_server_init', '4.4', [949], '5.0'],
            ['mysqli_result_current_field', '4.4', [950], '5.0'],
            ['mysqli_result_data_seek', '4.4', [951], '5.0'],
            ['mysqli_result_fetch_array', '4.4', [952], '5.0'],
            ['mysqli_result_fetch_assoc', '4.4', [953], '5.0'],
            ['mysqli_result_fetch_field_direct', '4.4', [954], '5.0'],
            ['mysqli_result_fetch_field', '4.4', [955], '5.0'],
            ['mysqli_result_fetch_fields', '4.4', [956], '5.0'],
            ['mysqli_result_fetch_object', '4.4', [957], '5.0'],
            ['mysqli_result_fetch_row', '4.4', [958], '5.0'],
            ['mysqli_result_field_count', '4.4', [959], '5.0'],
            ['mysqli_result_field_seek', '4.4', [960], '5.0'],
            ['mysqli_result_free', '4.4', [961], '5.0'],
            ['mysqli_result_lengths', '4.4', [962], '5.0'],
            ['mysqli_result_num_rows', '4.4', [963], '5.0'],

            ['interface_exists', '5.0.1', [737], '5.1', '5.0'],

            ['mysqli_set_charset', '5.0.4', [874], '5.1', '5.0'],

            ['date_default_timezone_get', '5.0', [548], '5.1'],
            ['date_default_timezone_set', '5.0', [549], '5.1'],
            ['strptime', '5.0', [563], '5.1'],
            ['fputcsv', '5.0', [570], '5.1'],
            ['imageconvolution', '5.0', [683], '5.1'],
            ['iterator_apply', '5.0', [35], '5.1'],
            ['readline_callback_handler_install', '5.0', [511], '5.1'],
            ['readline_callback_handler_remove', '5.0', [512], '5.1'],
            ['readline_callback_read_char', '5.0', [513], '5.1'],
            ['readline_on_new_line', '5.0', [514], '5.1'],
            ['readline_redisplay', '5.0', [515], '5.1'],
            ['posix_access', '5.0', [696], '5.1'],
            ['posix_mknod', '5.0', [697], '5.1'],
            ['time_sleep_until', '5.0', [699], '5.1'],
            ['stream_context_get_default', '5.0', [703], '5.1'],
            ['stream_filter_remove', '5.0', [704], '5.1'],
            ['stream_socket_enable_crypto', '5.0', [709], '5.1'],
            ['stream_socket_pair', '5.0', [710], '5.1'],
            ['stream_wrapper_restore', '5.0', [711], '5.1'],
            ['stream_wrapper_unregister', '5.0', [712], '5.1'],
            ['inet_ntop', '5.0', [721], '5.1'],
            ['inet_pton', '5.0', [722], '5.1'],
            ['apache_reset_timeout', '5.0', [729], '5.1'],
            ['htmlspecialchars_decode', '5.0', [731], '5.1'],
            ['array_diff_key', '5.0', [732], '5.1'],
            ['array_diff_ukey', '5.0', [733], '5.1'],
            ['array_intersect_key', '5.0', [734], '5.1'],
            ['array_intersect_ukey', '5.0', [735], '5.1'],
            ['array_product', '5.0', [736], '5.1'],
            ['property_exists', '5.0', [738], '5.1'],
            ['libxml_clear_errors', '5.0', [742], '5.1'],
            ['libxml_get_errors', '5.0', [743], '5.1'],
            ['libxml_get_last_error', '5.0', [744], '5.1'],
            ['libxml_set_streams_context', '5.0', [745], '5.1'],
            ['libxml_use_internal_errors', '5.0', [746], '5.1'],
            ['fbsql_rows_fetched', '5.0', [787], '5.1'],
            ['fbsql_set_characterset', '5.0', [788], '5.1'],
            ['mysqli_embedded_server_end', '5.0', [924], '5.1'],
            ['mysqli_embedded_server_start', '5.0', [925], '5.1'],
            ['mysqli_get_charset', '5.0', [855], '5.1'],
            ['mysqli_get_warnings', '5.0', [858], '5.1'],
            ['mysqli_stmt_get_warnings', '5.0', [902], '5.1'],
            ['pg_execute', '5.0', [972], '5.1'],
            ['pg_fetch_all_columns', '5.0', [973], '5.1'],
            ['pg_field_type_oid', '5.0', [974], '5.1'],
            ['pg_prepare', '5.0', [975], '5.1'],
            ['pg_query_params', '5.0', [976], '5.1'],
            ['pg_result_error_field', '5.0', [977], '5.1'],
            ['pg_send_execute', '5.0', [978], '5.1'],
            ['pg_send_prepare', '5.0', [979], '5.1'],
            ['pg_send_query_params', '5.0', [980], '5.1'],
            ['pg_set_error_verbosity', '5.0', [981], '5.1'],
            ['pg_transaction_status', '5.0', [982], '5.1'],

            ['date_sun_info', '5.1.1', [557], '5.2', '5.1'],
            ['hash_algos', '5.1.1', [521], '5.2', '5.1'],
            ['hash_file', '5.1.1', [522], '5.2', '5.1'],
            ['hash_final', '5.1.1', [523], '5.2', '5.1'],
            ['hash_hmac', '5.1.1', [442], '5.2', '5.1'],
            ['hash_hmac_file', '5.1.1', [524], '5.2', '5.1'],
            ['hash_init', '5.1.1', [525], '5.2', '5.1'],
            ['hash_update_file', '5.1.1', [526], '5.2', '5.1'],
            ['hash_update_stream', '5.1.1', [527], '5.2', '5.1'],
            ['hash_update', '5.1.1', [528], '5.2', '5.1'],
            ['hash', '5.1.1', [529], '5.2', '5.1'],
            ['xmlwriter_end_attribute', '5.1.1', [750], '5.2', '5.1'],
            ['xmlwriter_end_cdata', '5.1.1', [751], '5.2', '5.1'],
            ['xmlwriter_end_comment', '5.1.1', [752], '5.2', '5.1'],
            ['xmlwriter_end_document', '5.1.1', [753], '5.2', '5.1'],
            ['xmlwriter_end_dtd_attlist', '5.1.1', [754], '5.2', '5.1'],
            ['xmlwriter_end_dtd_element', '5.1.1', [755], '5.2', '5.1'],
            ['xmlwriter_end_dtd', '5.1.1', [756], '5.2', '5.1'],
            ['xmlwriter_end_element', '5.1.1', [757], '5.2', '5.1'],
            ['xmlwriter_end_pi', '5.1.1', [758], '5.2', '5.1'],
            ['xmlwriter_flush', '5.1.1', [759], '5.2', '5.1'],
            ['xmlwriter_open_memory', '5.1.1', [760], '5.2', '5.1'],
            ['xmlwriter_open_uri', '5.1.1', [761], '5.2', '5.1'],
            ['xmlwriter_output_memory', '5.1.1', [762], '5.2', '5.1'],
            ['xmlwriter_set_indent_string', '5.1.1', [763], '5.2', '5.1'],
            ['xmlwriter_set_indent', '5.1.1', [764], '5.2', '5.1'],
            ['xmlwriter_start_attribute_ns', '5.1.1', [765], '5.2', '5.1'],
            ['xmlwriter_start_attribute', '5.1.1', [766], '5.2', '5.1'],
            ['xmlwriter_start_cdata', '5.1.1', [767], '5.2', '5.1'],
            ['xmlwriter_start_comment', '5.1.1', [768], '5.2', '5.1'],
            ['xmlwriter_start_document', '5.1.1', [769], '5.2', '5.1'],
            ['xmlwriter_start_dtd_attlist', '5.1.1', [770], '5.2', '5.1'],
            ['xmlwriter_start_dtd_element', '5.1.1', [771], '5.2', '5.1'],
            ['xmlwriter_start_dtd', '5.1.1', [772], '5.2', '5.1'],
            ['xmlwriter_start_element_ns', '5.1.1', [773], '5.2', '5.1'],
            ['xmlwriter_start_element', '5.1.1', [774], '5.2', '5.1'],
            ['xmlwriter_start_pi', '5.1.1', [775], '5.2', '5.1'],
            ['xmlwriter_text', '5.1.1', [776], '5.2', '5.1'],
            ['xmlwriter_write_attribute_ns', '5.1.1', [777], '5.2', '5.1'],
            ['xmlwriter_write_attribute', '5.1.1', [778], '5.2', '5.1'],
            ['xmlwriter_write_cdata', '5.1.1', [779], '5.2', '5.1'],
            ['xmlwriter_write_comment', '5.1.1', [780], '5.2', '5.1'],
            ['xmlwriter_write_dtd_attlist', '5.1.1', [781], '5.2', '5.1'],
            ['xmlwriter_write_dtd_element', '5.1.1', [782], '5.2', '5.1'],
            ['xmlwriter_write_dtd', '5.1.1', [783], '5.2', '5.1'],
            ['xmlwriter_write_element_ns', '5.1.1', [784], '5.2', '5.1'],
            ['xmlwriter_write_element', '5.1.1', [785], '5.2', '5.1'],
            ['xmlwriter_write_pi', '5.1.1', [786], '5.2', '5.1'],

            ['oci_bind_array_by_name', '5.1.1', [964], '5.2', '5.1'],

            ['imap_savebody', '5.1.2', [688], '5.2', '5.1'],
            ['lchgrp', '5.1.2', [571], '5.2', '5.1'],
            ['lchown', '5.1.2', [572], '5.2', '5.1'],
            ['timezone_name_from_abbr', '5.1.2', [20], '5.2', '5.1'],
            ['sys_getloadavg', '5.1.2', [698], '5.2', '5.1'],
            ['curl_setopt_array', '5.1.2', [719], '5.2', '5.1'],

            ['array_fill_keys', '5.1', [12], '5.2'],
            ['array_fill_keys', '5.1', [13], '5.3'], // Test (global) namespaced function.
            ['error_get_last', '5.1', [14], '5.2'],
            ['image_type_to_extension', '5.1', [15], '5.2'],
            ['memory_get_peak_usage', '5.1', [16], '5.2'],
            ['date_create', '5.1', [546], '5.2'],
            ['date_date_set', '5.1', [547], '5.2'],
            ['date_format', '5.1', [550], '5.2'],
            ['date_isodate_set', '5.1', [553], '5.2'],
            ['date_modify', '5.1', [554], '5.2'],
            ['date_offset_get', '5.1', [555], '5.2'],
            ['date_parse', '5.1', [556], '5.2'],
            ['date_time_set', '5.1', [558], '5.2'],
            ['date_timezone_get', '5.1', [561], '5.2'],
            ['date_timezone_set', '5.1', [562], '5.2'],
            ['timezone_abbreviations_list', '5.1', [18], '5.2'],
            ['timezone_identifiers_list', '5.1', [19], '5.2'],
            ['timezone_open', '5.1', [564], '5.2'],
            ['timezone_name_get', '5.1', [566], '5.2'],
            ['timezone_offset_get', '5.1', [567], '5.2'],
            ['timezone_transitions_get', '5.1', [568], '5.2'],
            ['mb_stripos', '5.1', [25], '5.2'],
            ['mb_stristr', '5.1', [26], '5.2'],
            ['mb_strrchr', '5.1', [27], '5.2'],
            ['mb_strrichr', '5.1', [28], '5.2'],
            ['mb_strripos', '5.1', [29], '5.2'],
            ['mb_strstr', '5.1', [681], '5.2'],
            ['ming_setSWFCompression', '5.1', [30], '5.2'],
            ['openssl_csr_get_public_key', '5.1', [31], '5.2'],
            ['openssl_csr_get_subject', '5.1', [32], '5.2'],
            ['openssl_pkey_get_details', '5.1', [33], '5.2'],
            ['spl_object_hash', '5.1', [34], '5.2'],
            ['preg_last_error', '5.1', [36], '5.2'],
            ['pg_field_table', '5.1', [37], '5.2'],
            ['posix_initgroups', '5.1', [38], '5.2'],
            ['gmp_nextprime', '5.1', [39], '5.2'],
            ['xmlwriter_full_end_element', '5.1', [40], '5.2'],
            ['xmlwriter_write_raw', '5.1', [41], '5.2'],
            ['filter_has_var', '5.1', [45], '5.2'],
            ['filter_id', '5.1', [46], '5.2'],
            ['filter_input_array', '5.1', [47], '5.2'],
            ['filter_input', '5.1', [48], '5.2'],
            ['filter_list', '5.1', [49], '5.2'],
            ['filter_var_array', '5.1', [50], '5.2'],
            ['filter_var', '5.1', [51], '5.2'],
            ['json_decode', '5.1', [52], '5.2'],
            ['json_encode', '5.1', [53], '5.2'],
            ['zip_close', '5.1', [54], '5.2'],
            ['zip_entry_close', '5.1', [55], '5.2'],
            ['zip_entry_compressedsize', '5.1', [56], '5.2'],
            ['zip_entry_compressionmethod', '5.1', [57], '5.2'],
            ['zip_entry_filesize', '5.1', [58], '5.2'],
            ['zip_entry_name', '5.1', [59], '5.2'],
            ['zip_entry_open', '5.1', [60], '5.2'],
            ['zip_entry_read', '5.1', [61], '5.2'],
            ['zip_open', '5.1', [62], '5.2'],
            ['zip_read', '5.1', [63], '5.2'],
            ['stream_notification_callback', '5.1', [706], '5.2'],
            ['snmp_set_oid_output_format', '5.1', [723], '5.2'],
            ['snmp2_get', '5.1', [724], '5.2'],
            ['snmp2_getnext', '5.1', [725], '5.2'],
            ['snmp2_real_walk', '5.1', [726], '5.2'],
            ['snmp2_set', '5.1', [727], '5.2'],
            ['snmp2_walk', '5.1', [728], '5.2'],

            ['sys_get_temp_dir', '5.2.0', [17], '5.3', '5.2'],
            ['stream_socket_shutdown', '5.2.0', [21], '5.3', '5.2'],
            ['xmlwriter_start_dtd_entity', '5.2.0', [42], '5.3', '5.2'],
            ['xmlwriter_end_dtd_entity', '5.2.0', [43], '5.3', '5.2'],
            ['xmlwriter_write_dtd_entity', '5.2.0', [44], '5.3', '5.2'],

            ['imagegrabscreen', '5.2.1', [22], '5.3', '5.2'],
            ['imagegrabwindow', '5.2.1', [23], '5.3', '5.2'],
            ['openssl_pkcs12_export_to_file', '5.2.1', [539], '5.3', '5.2'],
            ['openssl_pkcs12_export', '5.2.1', [540], '5.3', '5.2'],
            ['openssl_pkcs12_read', '5.2.1', [541], '5.3', '5.2'],

            ['mysql_set_charset', '5.2.2', [830], '5.3', '5.2'],

            ['php_ini_loaded_file', '5.2.3', [509], '5.3', '5.2'],
            ['stream_is_local', '5.2.3', [705], '5.3', '5.2'],

            ['libxml_disable_entity_loader', '5.2.11', [24], '5.3', '5.2'],

            ['array_replace', '5.2', [65], '5.3'],
            ['array_replace_recursive', '5.2', [66], '5.3'],
            ['class_alias', '5.2', [67], '5.3'],
            ['forward_static_call', '5.2', [68], '5.3'],
            ['forward_static_call_array', '5.2', [69], '5.3'],
            ['gc_collect_cycles', '5.2', [70], '5.3'],
            ['gc_disable', '5.2', [71], '5.3'],
            ['gc_enable', '5.2', [72], '5.3'],
            ['gc_enabled', '5.2', [73], '5.3'],
            ['get_called_class', '5.2', [74], '5.3'],
            ['gethostname', '5.2', [75], '5.3'],
            ['header_remove', '5.2', [76], '5.3'],
            ['lcfirst', '5.2', [77], '5.3'],
            ['parse_ini_string', '5.2', [78], '5.3'],
            ['quoted_printable_encode', '5.2', [79], '5.3'],
            ['str_getcsv', '5.2', [80], '5.3'],
            ['stream_context_set_default', '5.2', [81], '5.3'],
            ['stream_supports_lock', '5.2', [82], '5.3'],
            ['stream_context_get_params', '5.2', [83], '5.3'],
            ['date_add', '5.2', [84], '5.3'],
            ['date_create_from_format', '5.2', [85], '5.3'],
            ['date_diff', '5.2', [86], '5.3'],
            ['date_get_last_errors', '5.2', [87], '5.3'],
            ['date_interval_create_from_date_string', '5.2', [551], '5.3'],
            ['date_interval_format', '5.2', [552], '5.3'],
            ['date_parse_from_format', '5.2', [88], '5.3'],
            ['date_sub', '5.2', [89], '5.3'],
            ['date_timestamp_get', '5.2', [559], '5.3'],
            ['date_timestamp_set', '5.2', [560], '5.3'],
            ['timezone_location_get', '5.2', [565], '5.3'],
            ['timezone_version_get', '5.2', [90], '5.3'],
            ['gmp_testbit', '5.2', [91], '5.3'],
            ['hash_copy', '5.2', [92], '5.3'],
            ['imap_gc', '5.2', [93], '5.3'],
            ['imap_utf8_to_mutf7', '5.2', [94], '5.3'],
            ['imap_mutf7_to_utf8', '5.2', [95], '5.3'],
            ['json_last_error', '5.2', [96], '5.3'],
            ['mysqli_fetch_all', '5.2', [97], '5.3'],
            ['mysqli_get_connection_status', '5.2', [98], '5.3'],
            ['mysqli_poll', '5.2', [99], '5.3'],
            ['mysqli_reap_async_query', '5.2', [100], '5.3'],
            ['mysqli_get_client_stats', '5.2', [857], '5.3'],
            ['mysqli_get_cache_stats', '5.2', [142], '5.3'],
            ['mysqli_get_connection_stats', '5.2', [856], '5.3'],
            ['mysqli_refresh', '5.2', [871], '5.3'],
            ['mysqli_result_fetch_all', '5.2', [947], '5.3'],
            ['mysqli_stmt_get_result', '5.2', [901], '5.3'],
            ['mysqli_stmt_more_results', '5.2', [903], '5.3'],
            ['mysqli_stmt_next_result', '5.2', [904], '5.3'],
            ['openssl_random_pseudo_bytes', '5.2', [101], '5.3'],
            ['pcntl_signal_dispatch', '5.2', [102], '5.3'],
            ['pcntl_sigprocmask', '5.2', [103], '5.3'],
            ['pcntl_sigtimedwait', '5.2', [104], '5.3'],
            ['pcntl_sigwaitinfo', '5.2', [105], '5.3'],
            ['preg_filter', '5.2', [106], '5.3'],
            ['msg_queue_exists', '5.2', [107], '5.3'],
            ['shm_has_vars', '5.2', [108], '5.3'],
            ['acosh', '5.2', [109], '5.3'],
            ['asinh', '5.2', [110], '5.3'],
            ['atanh', '5.2', [111], '5.3'],
            ['expm1', '5.2', [112], '5.3'],
            ['log1p', '5.2', [113], '5.3'],
            ['enchant_broker_describe', '5.2', [114], '5.3'],
            ['enchant_broker_dict_exists', '5.2', [115], '5.3'],
            ['enchant_broker_free_dict', '5.2', [116], '5.3'],
            ['enchant_broker_free', '5.2', [117], '5.3'],
            ['enchant_broker_get_error', '5.2', [118], '5.3'],
            ['enchant_broker_init', '5.2', [119], '5.3'],
            ['enchant_broker_list_dicts', '5.2', [120], '5.3'],
            ['enchant_broker_request_dict', '5.2', [121], '5.3'],
            ['enchant_broker_request_pwl_dict', '5.2', [122], '5.3'],
            ['enchant_broker_set_ordering', '5.2', [123], '5.3'],
            ['enchant_dict_add_to_personal', '5.2', [124], '5.3'],
            ['enchant_dict_add_to_session', '5.2', [125], '5.3'],
            ['enchant_dict_check', '5.2', [126], '5.3'],
            ['enchant_dict_describe', '5.2', [127], '5.3'],
            ['enchant_dict_get_error', '5.2', [128], '5.3'],
            ['enchant_dict_is_in_session', '5.2', [129], '5.3'],
            ['enchant_dict_quick_check', '5.2', [130], '5.3'],
            ['enchant_dict_store_replacement', '5.2', [131], '5.3'],
            ['enchant_dict_suggest', '5.2', [132], '5.3'],
            ['finfo_buffer', '5.2', [133], '5.3'],
            ['finfo_close', '5.2', [134], '5.3'],
            ['finfo_file', '5.2', [135], '5.3'],
            ['finfo_open', '5.2', [136], '5.3'],
            ['finfo_set_flags', '5.2', [137], '5.3'],
            ['collator_asort', '5.2', [579], '5.3'],
            ['collator_compare', '5.2', [580], '5.3'],
            ['collator_create', '5.2', [581], '5.3'],
            ['collator_get_attribute', '5.2', [582], '5.3'],
            ['collator_get_error_code', '5.2', [583], '5.3'],
            ['collator_get_error_message', '5.2', [584], '5.3'],
            ['collator_get_locale', '5.2', [585], '5.3'],
            ['collator_get_strength', '5.2', [587], '5.3'],
            ['collator_set_attribute', '5.2', [588], '5.3'],
            ['collator_set_strength', '5.2', [589], '5.3'],
            ['collator_sort_with_sort_keys', '5.2', [590], '5.3'],
            ['collator_sort', '5.2', [591], '5.3'],
            ['datefmt_create', '5.2', [645], '5.3'],
            ['datefmt_get_datetype', '5.2', [646], '5.3'],
            ['datefmt_get_timetype', '5.2', [647], '5.3'],
            ['datefmt_get_calendar', '5.2', [648], '5.3'],
            ['datefmt_set_calendar', '5.2', [649], '5.3'],
            ['datefmt_get_timezone_id', '5.2', [650], '5.3'],
            ['datefmt_set_timezone_id', '5.2', [651], '5.3'],
            ['datefmt_set_pattern', '5.2', [652], '5.3'],
            ['datefmt_get_pattern', '5.2', [653], '5.3'],
            ['datefmt_get_locale', '5.2', [654], '5.3'],
            ['datefmt_set_lenient', '5.2', [655], '5.3'],
            ['datefmt_is_lenient', '5.2', [656], '5.3'],
            ['datefmt_format', '5.2', [657], '5.3'],
            ['datefmt_parse', '5.2', [658], '5.3'],
            ['datefmt_localtime', '5.2', [659], '5.3'],
            ['datefmt_get_error_code', '5.2', [660], '5.3'],
            ['datefmt_get_error_message', '5.2', [661], '5.3'],
            ['grapheme_extract', '5.2', [668], '5.3'],
            ['grapheme_stripos', '5.2', [669], '5.3'],
            ['grapheme_stristr', '5.2', [670], '5.3'],
            ['grapheme_strlen', '5.2', [671], '5.3'],
            ['grapheme_strpos', '5.2', [672], '5.3'],
            ['grapheme_strripos', '5.2', [673], '5.3'],
            ['grapheme_strrpos', '5.2', [674], '5.3'],
            ['grapheme_strstr', '5.2', [675], '5.3'],
            ['grapheme_substr', '5.2', [676], '5.3'],
            ['idn_to_ascii', '5.2', [677], '5.3'],
            ['idn_to_utf8', '5.2', [678], '5.3'],
            ['intl_error_name', '5.2', [138], '5.3'],
            ['intl_get_error_code', '5.2', [139], '5.3'],
            ['intl_get_error_message', '5.2', [140], '5.3'],
            ['intl_is_failure', '5.2', [141], '5.3'],
            ['locale_accept_from_http', '5.2', [608], '5.3'],
            ['locale_canonicalize', '5.2', [609], '5.3'],
            ['locale_compose', '5.2', [610], '5.3'],
            ['locale_filter_matches', '5.2', [611], '5.3'],
            ['locale_get_all_variants', '5.2', [612], '5.3'],
            ['locale_get_default', '5.2', [613], '5.3'],
            ['locale_get_display_language', '5.2', [614], '5.3'],
            ['locale_get_display_name', '5.2', [615], '5.3'],
            ['locale_get_display_region', '5.2', [616], '5.3'],
            ['locale_get_display_script', '5.2', [617], '5.3'],
            ['locale_get_display_variant', '5.2', [618], '5.3'],
            ['locale_get_keywords', '5.2', [619], '5.3'],
            ['locale_get_primary_language', '5.2', [620], '5.3'],
            ['locale_get_region', '5.2', [621], '5.3'],
            ['locale_get_script', '5.2', [622], '5.3'],
            ['locale_lookup', '5.2', [623], '5.3'],
            ['locale_parse', '5.2', [624], '5.3'],
            ['locale_set_default', '5.2', [625], '5.3'],
            ['msgfmt_create', '5.2', [628], '5.3'],
            ['msgfmt_format_message', '5.2', [629], '5.3'],
            ['msgfmt_format', '5.2', [630], '5.3'],
            ['msgfmt_get_error_code', '5.2', [631], '5.3'],
            ['msgfmt_get_error_message', '5.2', [632], '5.3'],
            ['msgfmt_get_locale', '5.2', [633], '5.3'],
            ['msgfmt_get_pattern', '5.2', [634], '5.3'],
            ['msgfmt_parse_message', '5.2', [635], '5.3'],
            ['msgfmt_parse', '5.2', [636], '5.3'],
            ['msgfmt_set_pattern', '5.2', [637], '5.3'],
            ['normalizer_is_normalized', '5.2', [626], '5.3'],
            ['normalizer_normalize', '5.2', [627], '5.3'],
            ['numfmt_create', '5.2', [592], '5.3'],
            ['numfmt_format_currency', '5.2', [593], '5.3'],
            ['numfmt_format', '5.2', [594], '5.3'],
            ['numfmt_get_attribute', '5.2', [595], '5.3'],
            ['numfmt_get_error_code', '5.2', [596], '5.3'],
            ['numfmt_get_error_message', '5.2', [597], '5.3'],
            ['numfmt_get_locale', '5.2', [598], '5.3'],
            ['numfmt_get_pattern', '5.2', [599], '5.3'],
            ['numfmt_get_symbol', '5.2', [600], '5.3'],
            ['numfmt_get_text_attribute', '5.2', [601], '5.3'],
            ['numfmt_parse_currency', '5.2', [602], '5.3'],
            ['numfmt_parse', '5.2', [603], '5.3'],
            ['numfmt_set_attribute', '5.2', [604], '5.3'],
            ['numfmt_set_pattern', '5.2', [605], '5.3'],
            ['numfmt_set_symbol', '5.2', [606], '5.3'],
            ['numfmt_set_text_attribute', '5.2', [607], '5.3'],
            ['openssl_decrypt', '5.2', [532], '5.3'],
            ['openssl_dh_compute_key', '5.2', [533], '5.3'],
            ['openssl_digest', '5.2', [534], '5.3'],
            ['openssl_encrypt', '5.2', [535], '5.3'],
            ['openssl_get_cipher_methods', '5.2', [536], '5.3'],
            ['openssl_get_md_methods', '5.2', [538], '5.3'],
            ['mb_encoding_aliases', '5.2', [679], '5.3'],

            ['enchant_broker_get_dict_path', '5.3.0', [576], '5.4', '5.3'],
            ['enchant_broker_set_dict_path', '5.3.0', [577], '5.4', '5.3'],

            ['collator_get_sort_key', '5.3.1', [586], '5.4', '5.3'],
            ['resourcebundle_count', '5.3.1', [662], '5.4', '5.3'],
            ['resourcebundle_create', '5.3.1', [663], '5.4', '5.3'],
            ['resourcebundle_get_error_code', '5.3.1', [664], '5.4', '5.3'],
            ['resourcebundle_get_error_message', '5.3.1', [665], '5.4', '5.3'],
            ['resourcebundle_get', '5.3.1', [666], '5.4', '5.3'],
            ['resourcebundle_locales', '5.3.1', [667], '5.4', '5.3'],
            ['realpath_cache_get', '5.3.1', [573], '5.4', '5.3'],
            ['realpath_cache_size', '5.3.1', [574], '5.4', '5.3'],
            ['stream_resolve_include_path', '5.3.1', [707], '5.4', '5.3'],
            ['oci_set_action', '5.3.1', [965], '5.4', '5.3'],
            ['oci_set_client_info', '5.3.1', [966], '5.4', '5.3'],
            ['oci_set_client_identifier', '5.3.1', [967], '5.4', '5.3'],
            ['oci_set_edition', '5.3.1', [968], '5.4', '5.3'],
            ['oci_set_module_name', '5.3.1', [969], '5.4', '5.3'],

            ['openssl_cipher_iv_length', '5.3.2', [531], '5.4', '5.3'],
            ['stream_set_read_buffer', '5.3.2', [708], '5.4', '5.3'],
            ['fastcgi_finish_request', '5.3.2', [730], '5.4', '5.3'],

            ['pcntl_errno', '5.3.3', [693], '5.4', '5.3'],
            ['pcntl_get_last_error', '5.3.3', [694], '5.4', '5.3'],
            ['pcntl_strerror', '5.3.3', [695], '5.4', '5.3'],

            ['imap_fetchmime', '5.3.5', [687], '5.4', '5.3'],

            ['oci_client_version', '5.3.6', [970], '5.4', '5.3'],

            ['hex2bin', '5.3', [144], '5.4'],
            ['http_response_code', '5.3', [145], '5.4'],
            ['get_declared_traits', '5.3', [146], '5.4'],
            ['getimagesizefromstring', '5.3', [147], '5.4'],
            ['imagecreatefromwebp', '5.3', [684], '5.4'],
            ['imagewebp', '5.3', [686], '5.4'],
            ['stream_set_chunk_size', '5.3', [148], '5.4'],
            ['socket_import_stream', '5.3', [149], '5.4'],
            ['trait_exists', '5.3', [150], '5.4'],
            ['header_register_callback', '5.3', [151], '5.4'],
            ['class_uses', '5.3', [152], '5.4'],
            ['session_status', '5.3', [153], '5.4'],
            ['session_register_shutdown', '5.3', [154], '5.4'],
            ['mysqli_error_list', '5.3', [155], '5.4'],
            ['mysqli_stmt_error_list', '5.3', [156], '5.4'],
            ['libxml_set_external_entity_loader', '5.3', [157], '5.4'],
            ['ldap_control_paged_result', '5.3', [158], '5.4'],
            ['ldap_control_paged_result_response', '5.3', [159], '5.4'],
            ['transliterator_create', '5.3', [160], '5.4'],
            ['transliterator_create_from_rules', '5.3', [161], '5.4'],
            ['transliterator_create_inverse', '5.3', [162], '5.4'],
            ['transliterator_get_error_code', '5.3', [163], '5.4'],
            ['transliterator_get_error_message', '5.3', [164], '5.4'],
            ['transliterator_list_ids', '5.3', [165], '5.4'],
            ['transliterator_transliterate', '5.3', [166], '5.4'],
            ['zlib_decode', '5.3', [167], '5.4'],
            ['zlib_encode', '5.3', [168], '5.4'],
            ['gzdecode', '5.3', [517], '5.4'],

            ['mb_ereg_replace_callback', '5.4.0', [680], '5.5', '5.4'],

            ['pg_escape_literal', '5.4.3', [200], '5.5', '5.4'],
            ['pg_escape_identifier', '5.4.3', [201], '5.5', '5.4'],

            ['array_column', '5.4', [170], '5.5'],
            ['boolval', '5.4', [171], '5.5'],
            ['json_last_error_msg', '5.4', [172], '5.5'],
            ['password_get_info', '5.4', [173], '5.5'],
            ['password_hash', '5.4', [174], '5.5'],
            ['password_needs_rehash', '5.4', [175], '5.5'],
            ['password_verify', '5.4', [176], '5.5'],
            ['hash_pbkdf2', '5.4', [177], '5.5'],
            ['openssl_pbkdf2', '5.4', [178], '5.5'],
            ['curl_escape', '5.4', [179], '5.5'],
            ['curl_file_create', '5.4', [180], '5.5'],
            ['curl_multi_setopt', '5.4', [181], '5.5'],
            ['curl_multi_strerror', '5.4', [182], '5.5'],
            ['curl_pause', '5.4', [183], '5.5'],
            ['curl_reset', '5.4', [184], '5.5'],
            ['curl_share_close', '5.4', [185], '5.5'],
            ['curl_share_init', '5.4', [186], '5.5'],
            ['curl_share_setopt', '5.4', [187], '5.5'],
            ['curl_strerror', '5.4', [188], '5.5'],
            ['curl_unescape', '5.4', [189], '5.5'],
            ['date_create_immutable_from_format', '5.4', [544], '5.5'],
            ['date_create_immutable', '5.4', [545], '5.5'],
            ['imageaffine', '5.4', [682], '5.5'],
            ['imageaffinematrixconcat', '5.4', [190], '5.5'],
            ['imageaffinematrixget', '5.4', [191], '5.5'],
            ['imagecrop', '5.4', [192], '5.5'],
            ['imagecropauto', '5.4', [193], '5.5'],
            ['imageflip', '5.4', [194], '5.5'],
            ['imagepalettetotruecolor', '5.4', [195], '5.5'],
            ['imagescale', '5.4', [196], '5.5'],
            ['imagesetinterpolation', '5.4', [685], '5.5'],
            ['mysqli_begin_transaction', '5.4', [197], '5.5'],
            ['mysqli_release_savepoint', '5.4', [198], '5.5'],
            ['mysqli_savepoint', '5.4', [199], '5.5'],
            ['socket_sendmsg', '5.4', [202], '5.5'],
            ['socket_recvmsg', '5.4', [203], '5.5'],
            ['socket_cmsg_space', '5.4', [204], '5.5'],
            ['cli_get_process_title', '5.4', [205], '5.5'],
            ['cli_set_process_title', '5.4', [206], '5.5'],
            ['datefmt_format_object', '5.4', [207], '5.5'],
            ['datefmt_get_calendar_object', '5.4', [208, 211], '5.5'],
            ['datefmt_get_timezone', '5.4', [209], '5.5'],
            ['datefmt_set_timezone', '5.4', [210], '5.5'],
            ['intlcal_create_instance', '5.4', [212], '5.5'],
            ['intlcal_get_keyword_values_for_locale', '5.4', [213], '5.5'],
            ['intlcal_get_now', '5.4', [214], '5.5'],
            ['intlcal_get_available_locales', '5.4', [215], '5.5'],
            ['intlcal_get', '5.4', [216], '5.5'],
            ['intlcal_get_time', '5.4', [217], '5.5'],
            ['intlcal_set_time', '5.4', [218], '5.5'],
            ['intlcal_add', '5.4', [219], '5.5'],
            ['intlcal_set_time_zone', '5.4', [220], '5.5'],
            ['intlcal_after', '5.4', [221], '5.5'],
            ['intlcal_before', '5.4', [222], '5.5'],
            ['intlcal_set', '5.4', [223], '5.5'],
            ['intlcal_roll', '5.4', [224], '5.5'],
            ['intlcal_clear', '5.4', [225], '5.5'],
            ['intlcal_field_difference', '5.4', [226], '5.5'],
            ['intlcal_get_actual_maximum', '5.4', [227], '5.5'],
            ['intlcal_get_actual_minumum', '5.4', [228], '5.5'],
            ['intlcal_get_day_of_week_type', '5.4', [229], '5.5'],
            ['intlcal_get_first_day_of_week', '5.4', [230], '5.5'],
            ['intlcal_get_greatest_minimum', '5.4', [231], '5.5'],
            ['intlcal_get_least_maximum', '5.4', [232], '5.5'],
            ['intlcal_get_locale', '5.4', [233], '5.5'],
            ['intlcal_get_maximum', '5.4', [234], '5.5'],
            ['intlcal_get_minimal_days_in_first_week', '5.4', [235], '5.5'],
            ['intlcal_get_minimum', '5.4', [236], '5.5'],
            ['intlcal_get_time_zone', '5.4', [237], '5.5'],
            ['intlcal_get_type', '5.4', [238], '5.5'],
            ['intlcal_get_weekend_transition', '5.4', [239], '5.5'],
            ['intlcal_in_daylight_time', '5.4', [240], '5.5'],
            ['intlcal_is_equivalent_to', '5.4', [241], '5.5'],
            ['intlcal_is_lenient', '5.4', [242], '5.5'],
            ['intlcal_is_set', '5.4', [638], '5.5'],
            ['intlcal_is_weekend', '5.4', [639], '5.5'],
            ['intlcal_equals', '5.4', [243], '5.5'],
            ['intlcal_get_repeated_wall_time_option', '5.4', [244], '5.5'],
            ['intlcal_set_first_day_of_week', '5.4', [640], '5.5'],
            ['intlcal_set_lenient', '5.4', [641], '5.5'],
            ['intlcal_set_minimal_days_in_first_week', '5.4', [642], '5.5'],
            ['intlcal_get_skipped_wall_time_option', '5.4', [245], '5.5'],
            ['intlcal_set_repeated_wall_time_option', '5.4', [246], '5.5'],
            ['intlcal_set_skipped_wall_time_option', '5.4', [247], '5.5'],
            ['intlcal_from_date_time', '5.4', [248], '5.5'],
            ['intlcal_to_date_time', '5.4', [249], '5.5'],
            ['intlcal_get_error_code', '5.4', [250], '5.5'],
            ['intlcal_get_error_message', '5.4', [251], '5.5'],
            ['intlgregcal_create_instance', '5.4', [252], '5.5'],
            ['intlgregcal_set_gregorian_change', '5.4', [253], '5.5'],
            ['intlgregcal_get_gregorian_change', '5.4', [254], '5.5'],
            ['intlgregcal_is_leap_year', '5.4', [255], '5.5'],
            ['intltz_create_time_zone', '5.4', [256], '5.5'],
            ['intltz_create_default', '5.4', [257], '5.5'],
            ['intltz_get_id', '5.4', [258], '5.5'],
            ['intltz_get_gmt', '5.4', [259], '5.5'],
            ['intltz_get_unknown', '5.4', [260], '5.5'],
            ['intltz_create_enumeration', '5.4', [261], '5.5'],
            ['intltz_count_equivalent_ids', '5.4', [262], '5.5'],
            ['intltz_create_time_zone_id_enumeration', '5.4', [263], '5.5'],
            ['intltz_get_canonical_id', '5.4', [264], '5.5'],
            ['intltz_get_region', '5.4', [265], '5.5'],
            ['intltz_get_tz_data_version', '5.4', [266], '5.5'],
            ['intltz_get_equivalent_id', '5.4', [267], '5.5'],
            ['intltz_use_daylight_time', '5.4', [268], '5.5'],
            ['intltz_get_offset', '5.4', [269], '5.5'],
            ['intltz_get_raw_offset', '5.4', [270], '5.5'],
            ['intltz_has_same_rules', '5.4', [271], '5.5'],
            ['intltz_get_display_name', '5.4', [272], '5.5'],
            ['intltz_get_dst_savings', '5.4', [273], '5.5'],
            ['intltz_from_date_time_zone', '5.4', [274], '5.5'],
            ['intltz_to_date_time_zone', '5.4', [275], '5.5'],
            ['intltz_get_error_code', '5.4', [276], '5.5'],
            ['intltz_get_error_message', '5.4', [277], '5.5'],
            ['opcache_get_configuration', '5.4', [490], '5.5'],
            ['opcache_get_status', '5.4', [491], '5.5'],
            ['opcache_invalidate', '5.4', [492], '5.5'],
            ['opcache_reset', '5.4', [493], '5.5'],

            ['opcache_compile_file', '5.5.4', [489], '5.6', '5.5'],
            ['opcache_is_script_cached', '5.5.10', [494], '5.6', '5.5'],

            ['gmp_root', '5.5', [279], '5.6'],
            ['gmp_rootrem', '5.5', [280], '5.6'],
            ['hash_equals', '5.5', [281], '5.6'],
            ['ldap_escape', '5.5', [282], '5.6'],
            ['ldap_modify_batch', '5.4.25', [283], '5.6', '5.4'], // Function introduced in 5.4.25 and 5.5.10.
            ['mysqli_get_links_stats', '5.5', [284], '5.6'],
            ['openssl_get_cert_locations', '5.5', [285], '5.6'],
            ['openssl_x509_fingerprint', '5.5', [286], '5.6'],
            ['openssl_spki_new', '5.5', [287], '5.6'],
            ['openssl_spki_verify', '5.5', [288], '5.6'],
            ['openssl_spki_export_challenge', '5.5', [289], '5.6'],
            ['openssl_spki_export', '5.5', [290], '5.6'],
            ['phpdbg_clear', '5.5', [500], '5.6'],
            ['phpdbg_color', '5.5', [501], '5.6'],
            ['phpdbg_exec', '5.5', [503], '5.6'],
            ['phpdbg_prompt', '5.5', [505], '5.6'],
            ['pg_connect_poll', '5.5', [291], '5.6'],
            ['pg_consume_input', '5.5', [292], '5.6'],
            ['pg_flush', '5.5', [293], '5.6'],
            ['pg_socket', '5.5', [294], '5.6'],
            ['session_abort', '5.5', [295], '5.6'],
            ['session_reset', '5.5', [296], '5.6'],
            ['oci_get_implicit_resultset', '5.5', [971], '5.6'],

            ['gmp_export', '5.6.0', [689], '7.0', '5.6'],
            ['gmp_import', '5.6.0', [690], '7.0', '5.6'],

            ['gmp_random_bits', '5.6.2', [691], '7.0', '5.6'],
            ['gmp_random_range', '5.6.2', [692], '7.0', '5.6'],
            ['phpdbg_break_file', '5.6.2', [496], '7.0', '5.6'],
            ['phpdbg_break_function', '5.6.2', [497], '7.0', '5.6'],
            ['phpdbg_break_method', '5.6.2', [498], '7.0', '5.6'],
            ['phpdbg_break_next', '5.6.2', [499], '7.0', '5.6'],

            ['random_bytes', '5.6', [298], '7.0'],
            ['random_int', '5.6', [299], '7.0'],
            ['error_clear_last', '5.6', [300], '7.0'],
            ['gmp_random_seed', '5.6', [301], '7.0'],
            ['intdiv', '5.6', [302], '7.0'],
            ['preg_replace_callback_array', '5.6', [303], '7.0'],
            ['gc_mem_caches', '5.6', [304], '7.0'],
            ['get_resources', '5.6', [305], '7.0'],
            ['phpdbg_end_oplog', '5.6', [502], '7.0'],
            ['phpdbg_get_executable', '5.6', [504], '7.0'],
            ['phpdbg_start_oplog', '5.6', [506], '7.0'],
            ['posix_setrlimit', '5.6', [306], '7.0'],
            ['inflate_add', '5.6', [307], '7.0'],
            ['deflate_add', '5.6', [308], '7.0'],
            ['inflate_init', '5.6', [309], '7.0'],
            ['deflate_init', '5.6', [310], '7.0'],

            ['socket_export_stream', '7.0.6', [312], '7.1', '7.0'],

            ['curl_multi_errno', '7.0', [314], '7.1'],
            ['curl_share_errno', '7.0', [315], '7.1'],
            ['curl_share_strerror', '7.0', [316], '7.1'],
            ['is_iterable', '7.0', [317], '7.1'],
            ['pcntl_async_signals', '7.0', [318], '7.1'],
            ['pcntl_signal_get_handler', '7.0', [467], '7.1'],
            ['session_create_id', '7.0', [319], '7.1'],
            ['session_gc', '7.0', [320], '7.1'],
            ['sapi_windows_cp_set', '7.0', [462], '7.1'],
            ['sapi_windows_cp_get', '7.0', [463], '7.1'],
            ['sapi_windows_cp_is_utf8', '7.0', [464], '7.1'],
            ['sapi_windows_cp_conv', '7.0', [465], '7.1'],
            ['openssl_get_curve_names', '7.0', [537], '7.1'],
            ['intltz_get_id_for_windows_id', '7.0', [643], '7.1'],
            ['intltz_get_windows_id', '7.0', [644], '7.1'],

            ['hash_hkdf', '7.1.1', [466], '7.2', '7.1'],
            ['oci_register_taf_callback', '7.1.6', [322], '7.2', '7.1'],
            ['oci_unregister_taf_callback', '7.1.8', [323], '7.2', '7.1'],

            ['ldap_exop', '7.1', [325], '7.2'],
            ['ldap_exop_passwd', '7.1', [326], '7.2'],
            ['ldap_exop_whoami', '7.1', [327], '7.2'],
            ['ldap_parse_exop', '7.1', [328], '7.2'],
            ['socket_addrinfo_lookup', '7.1', [329], '7.2'],
            ['socket_addrinfo_connect', '7.1', [330], '7.2'],
            ['socket_addrinfo_bind', '7.1', [331], '7.2'],
            ['socket_addrinfo_explain', '7.1', [332], '7.2'],
            ['stream_isatty', '7.1', [333], '7.2'],
            ['sapi_windows_vt100_support', '7.1', [334], '7.2'],
            ['ftp_append', '7.1', [335], '7.2'],
            ['ftp_mlsd', '7.1', [720], '7.2'],
            ['imagebmp', '7.1', [336], '7.2'],
            ['imagecreatefrombmp', '7.1', [337], '7.2'],
            ['imagegetclip', '7.1', [338], '7.2'],
            ['imageopenpolygon', '7.1', [339], '7.2'],
            ['imageresolution', '7.1', [340], '7.2'],
            ['imagesetclip', '7.1', [341], '7.2'],
            ['inflate_get_read_len', '7.1', [518], '7.2'],
            ['inflate_get_status', '7.1', [519], '7.2'],
            ['hash_hmac_algos', '7.1', [342], '7.2'],
            ['mb_chr', '7.1', [343], '7.2'],
            ['mb_ord', '7.1', [344], '7.2'],
            ['mb_scrub', '7.1', [345], '7.2'],
            ['sodium_add', '7.1', [346], '7.2'],
            ['sodium_base642bin', '7.1', [421], '7.2'],
            ['sodium_bin2base64', '7.1', [422], '7.2'],
            ['sodium_bin2hex', '7.1', [347], '7.2'],
            ['sodium_compare', '7.1', [348], '7.2'],
            ['sodium_crypto_aead_aes256gcm_decrypt', '7.1', [349], '7.2'],
            ['sodium_crypto_aead_aes256gcm_encrypt', '7.1', [350], '7.2'],
            ['sodium_crypto_aead_aes256gcm_is_available', '7.1', [351], '7.2'],
            ['sodium_crypto_aead_aes256gcm_keygen', '7.1', [352], '7.2'],
            ['sodium_crypto_aead_chacha20poly1305_decrypt', '7.1', [353], '7.2'],
            ['sodium_crypto_aead_chacha20poly1305_encrypt', '7.1', [354], '7.2'],
            ['sodium_crypto_aead_chacha20poly1305_ietf_decrypt', '7.1', [355], '7.2'],
            ['sodium_crypto_aead_chacha20poly1305_ietf_encrypt', '7.1', [356], '7.2'],
            ['sodium_crypto_aead_chacha20poly1305_ietf_keygen', '7.1', [357], '7.2'],
            ['sodium_crypto_aead_chacha20poly1305_keygen', '7.1', [358], '7.2'],
            ['sodium_crypto_aead_xchacha20poly1305_ietf_decrypt', '7.1', [423], '7.2'],
            ['sodium_crypto_aead_xchacha20poly1305_ietf_encrypt', '7.1', [424], '7.2'],
            ['sodium_crypto_aead_xchacha20poly1305_ietf_keygen', '7.1', [425], '7.2'],
            ['sodium_crypto_auth_keygen', '7.1', [359], '7.2'],
            ['sodium_crypto_auth_verify', '7.1', [360], '7.2'],
            ['sodium_crypto_auth', '7.1', [361], '7.2'],
            ['sodium_crypto_box_keypair_from_secretkey_and_publickey', '7.1', [362], '7.2'],
            ['sodium_crypto_box_keypair', '7.1', [363], '7.2'],
            ['sodium_crypto_box_open', '7.1', [364], '7.2'],
            ['sodium_crypto_box_publickey_from_secretkey', '7.1', [365], '7.2'],
            ['sodium_crypto_box_publickey', '7.1', [366], '7.2'],
            ['sodium_crypto_box_seal_open', '7.1', [367], '7.2'],
            ['sodium_crypto_box_seal', '7.1', [368], '7.2'],
            ['sodium_crypto_box_secretkey', '7.1', [369], '7.2'],
            ['sodium_crypto_box_seed_keypair', '7.1', [370], '7.2'],
            ['sodium_crypto_box', '7.1', [371], '7.2'],
            ['sodium_crypto_generichash_final', '7.1', [372], '7.2'],
            ['sodium_crypto_generichash_init', '7.1', [373], '7.2'],
            ['sodium_crypto_generichash_keygen', '7.1', [374], '7.2'],
            ['sodium_crypto_generichash_update', '7.1', [375], '7.2'],
            ['sodium_crypto_generichash', '7.1', [376], '7.2'],
            ['sodium_crypto_kdf_derive_from_key', '7.1', [377], '7.2'],
            ['sodium_crypto_kdf_keygen', '7.1', [378], '7.2'],
            ['sodium_crypto_kx_client_session_keys', '7.1', [379], '7.2'],
            ['sodium_crypto_kx_keypair', '7.1', [380], '7.2'],
            ['sodium_crypto_kx_publickey', '7.1', [381], '7.2'],
            ['sodium_crypto_kx_secretkey', '7.1', [382], '7.2'],
            ['sodium_crypto_kx_seed_keypair', '7.1', [383], '7.2'],
            ['sodium_crypto_kx_server_session_keys', '7.1', [384], '7.2'],
            ['sodium_crypto_pwhash_scryptsalsa208sha256_str_verify', '7.1', [385], '7.2'],
            ['sodium_crypto_pwhash_scryptsalsa208sha256_str', '7.1', [386], '7.2'],
            ['sodium_crypto_pwhash_scryptsalsa208sha256', '7.1', [387], '7.2'],
            ['sodium_crypto_pwhash_str_needs_rehash', '7.1', [426], '7.2'],
            ['sodium_crypto_pwhash_str_verify', '7.1', [388], '7.2'],
            ['sodium_crypto_pwhash_str', '7.1', [389], '7.2'],
            ['sodium_crypto_pwhash', '7.1', [390], '7.2'],
            ['sodium_crypto_scalarmult_base', '7.1', [391], '7.2'],
            ['sodium_crypto_scalarmult', '7.1', [392], '7.2'],
            ['sodium_crypto_secretbox_keygen', '7.1', [393], '7.2'],
            ['sodium_crypto_secretbox_open', '7.1', [394], '7.2'],
            ['sodium_crypto_secretbox', '7.1', [395], '7.2'],
            ['sodium_crypto_secretstream_xchacha20poly1305_init_pull', '7.1', [427], '7.2'],
            ['sodium_crypto_secretstream_xchacha20poly1305_init_push', '7.1', [428], '7.2'],
            ['sodium_crypto_secretstream_xchacha20poly1305_keygen', '7.1', [429], '7.2'],
            ['sodium_crypto_secretstream_xchacha20poly1305_pull', '7.1', [430], '7.2'],
            ['sodium_crypto_secretstream_xchacha20poly1305_push', '7.1', [431], '7.2'],
            ['sodium_crypto_secretstream_xchacha20poly1305_rekey', '7.1', [432], '7.2'],
            ['sodium_crypto_shorthash_keygen', '7.1', [396], '7.2'],
            ['sodium_crypto_shorthash', '7.1', [397], '7.2'],
            ['sodium_crypto_sign_detached', '7.1', [398], '7.2'],
            ['sodium_crypto_sign_ed25519_pk_to_curve25519', '7.1', [399], '7.2'],
            ['sodium_crypto_sign_ed25519_sk_to_curve25519', '7.1', [400], '7.2'],
            ['sodium_crypto_sign_keypair_from_secretkey_and_publickey', '7.1', [401], '7.2'],
            ['sodium_crypto_sign_keypair', '7.1', [402], '7.2'],
            ['sodium_crypto_sign_open', '7.1', [403], '7.2'],
            ['sodium_crypto_sign_publickey_from_secretkey', '7.1', [404], '7.2'],
            ['sodium_crypto_sign_publickey', '7.1', [405], '7.2'],
            ['sodium_crypto_sign_secretkey', '7.1', [406], '7.2'],
            ['sodium_crypto_sign_seed_keypair', '7.1', [407], '7.2'],
            ['sodium_crypto_sign_verify_detached', '7.1', [408], '7.2'],
            ['sodium_crypto_sign', '7.1', [409], '7.2'],
            ['sodium_crypto_stream_keygen', '7.1', [410], '7.2'],
            ['sodium_crypto_stream_xor', '7.1', [411], '7.2'],
            ['sodium_crypto_stream', '7.1', [412], '7.2'],
            ['sodium_hex2bin', '7.1', [413], '7.2'],
            ['sodium_increment', '7.1', [414], '7.2'],
            ['sodium_memcmp', '7.1', [415], '7.2'],
            ['sodium_memzero', '7.1', [416], '7.2'],
            ['sodium_pad', '7.1', [417], '7.2'],
            ['sodium_unpad', '7.1', [418], '7.2'],
            ['spl_object_id', '7.1', [419], '7.2'],
            ['openssl_pkcs7_read', '7.1', [542], '7.2'],

            ['pg_lo_truncate', '5.5', [420], '5.6'],

            ['iterator_count', '5.0', [434], '5.1'],
            ['iterator_to_array', '5.0', [435], '5.1'],
            ['spl_autoload_call', '5.0', [436], '5.1'],
            ['spl_autoload_extensions', '5.0', [437], '5.1'],
            ['spl_autoload_functions', '5.0', [438], '5.1'],
            ['spl_autoload_register', '5.0', [439], '5.1'],
            ['spl_autoload_unregister', '5.0', [440], '5.1'],
            ['spl_autoload', '5.0', [441], '5.1'],

            ['hrtime', '7.2', [444], '7.3'],
            ['is_countable', '7.2', [445], '7.3'],
            ['array_key_first', '7.2', [446], '7.3'],
            ['array_key_last', '7.2', [447], '7.3'],
            ['fpm_get_status', '7.2', [448], '7.3'],
            ['net_get_interfaces', '7.2', [449], '7.3'],
            ['gmp_binomial', '7.2', [450], '7.3'],
            ['gmp_lcm', '7.2', [451], '7.3'],
            ['gmp_perfect_power', '7.2', [452], '7.3'],
            ['gmp_kronecker', '7.2', [453], '7.3'],
            ['ldap_exop_refresh', '7.2', [454], '7.3'],
            ['normalizer_get_raw_decomposition', '7.2', [455], '7.3'],
            ['openssl_pkey_derive', '7.2', [456], '7.3'],
            ['socket_wsaprotocol_info_export', '7.2', [458], '7.3'],
            ['socket_wsaprotocol_info_import', '7.2', [459], '7.3'],
            ['socket_wsaprotocol_info_release', '7.2', [460], '7.3'],
            ['gc_status', '7.2', [508], '7.3'],

            ['ldap_add_ext', '7.2', [469], '7.3'],
            ['ldap_bind_ext', '7.2', [470], '7.3'],
            ['ldap_delete_ext', '7.2', [471], '7.3'],
            ['ldap_mod_add_ext', '7.2', [472], '7.3'],
            ['ldap_mod_replace_ext', '7.2', [473], '7.3'],
            ['ldap_mod_del_ext', '7.2', [474], '7.3'],
            ['ldap_rename_ext', '7.2', [475], '7.3'],

            ['oci_set_call_timeout', '7.2.13', [477], '7.3', '7.2'],
            ['oci_set_db_operation', '7.2.13', [478], '7.3', '7.2'],

            ['mb_str_split', '7.3', [480], '7.4'],
            ['get_mangled_object_vars', '7.3', [481], '7.4'],
            ['openssl_x509_verify', '7.3', [482], '7.4'],
            ['pcntl_unshare', '7.3', [483], '7.4'],
            ['sapi_windows_set_ctrl_handler', '7.3', [484], '7.4'],
            ['sapi_windows_generate_ctrl_event', '7.3', [485], '7.4'],
            ['password_algos', '7.3', [486], '7.4'],
            ['imagecreatefromtga', '7.3', [487], '7.4'],

            ['fdiv', '7.4', [989], '8.0'],
            ['get_debug_type', '7.4', [990], '8.0'],
            ['get_resource_id', '7.4', [991], '8.0'],
            ['preg_last_error_msg', '7.4', [992], '8.0'],
            ['str_contains', '7.4', [993], '8.0'],
            ['str_ends_with', '7.4', [994], '8.0'],
            ['str_starts_with', '7.4', [995], '8.0'],
            ['imagegetinterpolation', '7.4', [996], '8.0'],
            ['enchant_dict_add', '7.4', [997], '8.0'],
            ['enchant_dict_is_added', '7.4', [998], '8.0'],
            ['ldap_count_references', '7.4', [999], '8.0'],
            ['openssl_cms_encrypt', '7.4', [984], '8.0'],
            ['openssl_cms_decrypt', '7.4', [985], '8.0'],
            ['openssl_cms_read', '7.4', [986], '8.0'],
            ['openssl_cms_sign', '7.4', [987], '8.0'],
            ['openssl_cms_verify', '7.4', [988], '8.0'],
        ];
    }


    /**
     * Test functions that shouldn't be flagged by this sniff.
     *
     * These are either userland methods or namespaced functions.
     *
     * @dataProvider dataNoFalsePositives
     *
     * @param int $line The line number.
     *
     * @return void
     */
    public function testNoFalsePositives($line)
    {
        $file = $this->sniffFile(__FILE__, '5.1'); // Low version below the first addition.
        $this->assertNoViolation($file, $line);
    }

    /**
     * Data provider.
     *
     * @see testNoFalsePositives()
     *
     * @return array
     */
    public function dataNoFalsePositives()
    {
        return [
            [4],
            [5],
            [6],
            [7],
        ];
    }


    /**
     * Verify no notices are thrown at all.
     *
     * @return void
     */
    public function testNoViolationsInFileOnValidVersion()
    {
        $file = $this->sniffFile(__FILE__, '99.0'); // High version beyond newest addition.
        $this->assertNoViolation($file);
    }
}
