<?php

/**
 * This file is part of the Nette Framework (https://nette.org)
 * Copyright (c) 2004 David Grudl (https://davidgrudl.com)
 */

declare(strict_types=1);

namespace Nette\PhpGenerator\Traits;

use Nette;
use Nette\PhpGenerator\Method;


/**
 * @internal
 */
trait MethodsAware
{
	/** @var array<string, Method> */
	private array $methods = [];


	/** @param  Method[]  $methods */
	public function setMethods(array $methods): static
	{
		(function (Method ...$methods) {})(...$methods);
		$this->methods = [];
		foreach ($methods as $m) {
			$this->methods[strtolower($m->getName())] = $m;
		}

		return $this;
	}


	/** @return Method[] */
	public function getMethods(): array
	{
		$res = [];
		foreach ($this->methods as $m) {
			$res[$m->getName()] = $m;
		}

		return $res;
	}


	public function getMethod(string $name): Method
	{
		return $this->methods[strtolower($name)] ?? throw new Nette\InvalidArgumentException("Method '$name' not found.");
	}


	public function addMethod(string $name): Method
	{
		$lower = strtolower($name);
		if (isset($this->methods[$lower])) {
			throw new Nette\InvalidStateException("Cannot add method '$name', because it already exists.");
		}
		$method = new Method($name);
		if (!$this->isInterface()) {
			$method->setPublic();
		}

		return $this->methods[$lower] = $method;
	}


	public function removeMethod(string $name): static
	{
		unset($this->methods[strtolower($name)]);
		return $this;
	}


	public function hasMethod(string $name): bool
	{
		return isset($this->methods[strtolower($name)]);
	}


	/**
	 * Inherits method from parent class or interface.
	 */
	public function inheritMethod(string $name, bool $callParent = false, bool $returnIfExists = false): Method
	{
		$lower = strtolower($name);
		if (isset($this->methods[$lower])) {
			return $returnIfExists
				? $this->methods[$lower]
				: throw new Nette\InvalidStateException("Cannot inherit method '$name', because it already exists.");
		}

		$parents = match (true) {
			$this instanceof Nette\PhpGenerator\ClassType => [...(array) $this->getExtends(), ...$this->getImplements()],
			$this instanceof Nette\PhpGenerator\InterfaceType => $this->getExtends(),
			$this instanceof Nette\PhpGenerator\EnumType => $this->getImplements(),
		};
		if (!$parents) {
			throw new Nette\InvalidStateException("Class '{$this->getName()}' has neither setExtends() nor setImplements() set.");
		}

		foreach ($parents as $parent) {
			try {
				$rm = new \ReflectionMethod($parent, $name);
			} catch (\ReflectionException) {
				continue;
			}
			$method = Method::from([$parent, $name]);
			if ($callParent) {
				$params = array_map(fn(\ReflectionParameter $param) => '$' . $param->getName(), $rm->getParameters());
				if ($rm->isVariadic()) {
					$params[] = '...' . array_pop($params);
				}
				$method->setBody('parent::' . $rm->getName() . '(' . implode(', ', $params) . ');');
			}
			return $this->methods[$lower] = $method;
		}

		throw new Nette\InvalidStateException("Method '$name' has not been found in any ancestor: " . implode(', ', $parents));
	}
}
