<?php

/**
 * This file is part of the Nette Framework (https://nette.org)
 * Copyright (c) 2004 David Grudl (https://davidgrudl.com)
 */

declare(strict_types=1);

namespace Nette\PhpGenerator\Traits;

use Nette;
use Nette\PhpGenerator\Modifier;
use Nette\PhpGenerator\PropertyHook;


/**
 * @internal
 */
trait PropertyLike
{
	/** @var array{'set' => ?string, 'get' => ?string} */
	private array $visibility = ['set' => null, 'get' => null];
	private bool $readOnly = false;

	/** @var array<string, ?PropertyHook> */
	private array $hooks = ['set' => null, 'get' => null];


	/**
	 * @param  ?string  $get  public|protected|private
	 * @param  ?string  $set  public|protected|private
	 */
	public function setVisibility(?string $get, ?string $set = null): static
	{
		if (!in_array($get, [Modifier::Public, Modifier::Protected, Modifier::Private, null], true)
			|| !in_array($set, [Modifier::Public, Modifier::Protected, Modifier::Private, null], true)) {
			throw new Nette\InvalidArgumentException('Argument must be public|protected|private.');
		}

		$this->visibility = ['set' => $set, 'get' => $get];
		return $this;
	}


	/** @param  'set'|'get'  $mode */
	public function getVisibility(string $mode = 'get'): ?string
	{
		return $this->visibility[$this->checkMode($mode)];
	}


	/** @param  'set'|'get'  $mode */
	public function setPublic(string $mode = 'get'): static
	{
		$this->visibility[$this->checkMode($mode)] = Modifier::Public;
		return $this;
	}


	/** @param  'set'|'get'  $mode */
	public function isPublic(string $mode = 'get'): bool
	{
		return in_array($this->visibility[$this->checkMode($mode)], [Modifier::Public, null], true);
	}


	/** @param  'set'|'get'  $mode */
	public function setProtected(string $mode = 'get'): static
	{
		$this->visibility[$this->checkMode($mode)] = Modifier::Protected;
		return $this;
	}


	/** @param  'set'|'get'  $mode */
	public function isProtected(string $mode = 'get'): bool
	{
		return $this->visibility[$this->checkMode($mode)] === Modifier::Protected;
	}


	/** @param  'set'|'get'  $mode */
	public function setPrivate(string $mode = 'get'): static
	{
		$this->visibility[$this->checkMode($mode)] = Modifier::Private;
		return $this;
	}


	/** @param  'set'|'get'  $mode */
	public function isPrivate(string $mode = 'get'): bool
	{
		return $this->visibility[$this->checkMode($mode)] === Modifier::Private;
	}


	public function setReadOnly(bool $state = true): static
	{
		$this->readOnly = $state;
		return $this;
	}


	public function isReadOnly(): bool
	{
		return $this->readOnly;
	}


	/**
	 * Replaces all hooks
	 * @param  PropertyHook[]  $hooks
	 */
	public function setHooks(array $hooks): static
	{
		(function (PropertyHook ...$hooks) {})(...$hooks);
		$this->hooks = $hooks;
		return $this;
	}


	/** @return array<string, PropertyHook> */
	public function getHooks(): array
	{
		return array_filter($this->hooks);
	}


	/** @param  'set'|'get'|\PropertyHookType  $type */
	public function addHook(string|\PropertyHookType $type, string $body = '', bool $short = false): PropertyHook
	{
		$type = is_string($type) ? $type : $type->value;
		return $this->hooks[$type] = (new PropertyHook)
			->setBody($body, short: $short);
	}


	/** @param  'set'|'get'|\PropertyHookType  $type */
	public function getHook(string|\PropertyHookType $type): ?PropertyHook
	{
		$type = is_string($type) ? $type : $type->value;
		return $this->hooks[$type] ?? null;
	}


	/** @param  'set'|'get'|\PropertyHookType  $type */
	public function hasHook(string|\PropertyHookType $type): bool
	{
		return isset($this->hooks[$type]);
	}


	private function checkMode(string $mode): string
	{
		return $mode === Modifier::Set || $mode === Modifier::Get
			? $mode
			: throw new Nette\InvalidArgumentException('Argument must be set|get.');
	}
}
