<?php

/**
 * This file is part of the Nette Framework (https://nette.org)
 * Copyright (c) 2004 David Grudl (https://davidgrudl.com)
 */

declare(strict_types=1);

namespace Nette\Neon;


/** @internal */
final class TokenStream
{
	private int $index = 0;


	public function __construct(
		/** @var Token[] */
		public /*readonly*/ array $tokens,
	) {
	}


	public function getIndex(): int
	{
		return $this->index;
	}


	public function seek(int $index): void
	{
		$this->index = $index;
	}


	public function isNext(int|string ...$types): bool
	{
		while (in_array($this->tokens[$this->index]->type, [Token::Comment, Token::Whitespace], strict: true)) {
			$this->index++;
		}

		return $types
			? in_array($this->tokens[$this->index]->type, $types, strict: true)
			: $this->tokens[$this->index]->type !== Token::End;
	}


	public function consume(int|string ...$types): ?Token
	{
		return $this->isNext(...$types)
			? $this->tokens[$this->index++]
			: null;
	}


	public function getIndentation(): string
	{
		return in_array($this->tokens[$this->index - 2]->type ?? null, [Token::Newline, null], strict: true)
			&& ($this->tokens[$this->index - 1]->type ?? null) === Token::Whitespace
			? $this->tokens[$this->index - 1]->text
			: '';
	}


	/** @return never */
	public function error(?string $message = null, ?int $pos = null): void
	{
		$pos ??= $this->index;
		$token = $this->tokens[$pos];
		$message ??= 'Unexpected ' . ($token->type === Token::End
			? 'end'
			: "'" . str_replace("\n", '<new line>', substr($token->text, 0, 40)) . "'");
		throw new Exception("$message $token->position.");
	}
}
