<?php

/**
 * This file is part of the Nette Framework (https://nette.org)
 * Copyright (c) 2004 David Grudl (https://davidgrudl.com)
 */

declare(strict_types=1);

namespace Nette\DI;

use Nette;
use Nette\DI\Definitions\Definition;
use Nette\DI\Definitions\Reference;
use Nette\DI\Definitions\Statement;
use Nette\PhpGenerator\Helpers as PhpHelpers;
use Nette\Utils\Strings;
use Nette\Utils\Validators;
use ReflectionClass;


/**
 * Services resolver
 * @internal
 */
class Resolver
{
	use Nette\SmartObject;

	/** @var ContainerBuilder */
	private $builder;

	/** @var Definition|null */
	private $currentService;

	/** @var \SplObjectStorage  circular reference detector */
	private $recursive;


	public function __construct(ContainerBuilder $builder)
	{
		$this->builder = $builder;
		$this->recursive = new \SplObjectStorage;
	}


	public function resolveDefinition(Definition $def): void
	{
		if ($this->recursive->contains($def)) {
			$names = array_map(function ($item) { return $item->getName(); }, iterator_to_array($this->recursive));
			throw new ServiceCreationException(sprintf('Circular reference detected for services: %s.', implode(', ', $names)));
		}

		try {
			$this->recursive->attach($def);

			$def->resolveType($this);

			if (!$def->getType() && $def->getAutowired()) {
				throw new ServiceCreationException('Unknown type, declare return type of factory method (for PHP 5 use annotation @return)');
			}
		} catch (\Exception $e) {
			throw $this->completeException($e, $def);

		} finally {
			$this->recursive->detach($def);
		}
	}


	private function getDefinition($name): Definition
	{
		return $name === ContainerBuilder::THIS_SERVICE
			? $this->currentService
			: $this->builder->getDefinition($name);
	}


	public function resolveServiceType(string $name): ?string
	{
		$def = $this->getDefinition($name);
		if (!$def->getType()) {
			$this->resolveDefinition($def);
		}
		return $def->getType();
	}


	private function resolveReferenceType(Reference $reference): ?string
	{
		return $reference->isName()
			? $this->resolveServiceType($reference->getValue())
			: $reference->getValue();
	}


	public function resolveEntityType($entity): ?string
	{
		$entity = $this->normalizeEntity($entity instanceof Statement ? $entity->getEntity() : $entity);

		if (is_array($entity)) {
			if ($entity[0] instanceof Reference || $entity[0] instanceof Statement) {
				$entity[0] = $this->resolveEntityType($entity[0]);
				if (!$entity[0]) {
					return null;
				}
			}

			try {
				$reflection = Nette\Utils\Callback::toReflection($entity[0] === '' ? $entity[1] : $entity);
				$refClass = $reflection instanceof \ReflectionMethod ? $reflection->getDeclaringClass() : null;
			} catch (\ReflectionException $e) {
			}

			if (isset($e) || ($refClass && (!$reflection->isPublic()
				|| ($refClass->isTrait() && !$reflection->isStatic())
			))) {
				throw new ServiceCreationException(sprintf('Method %s() is not callable.', Nette\Utils\Callback::toString($entity)), 0, $e ?? null);
			}
			$this->addDependency($reflection);

			$type = Helpers::getReturnType($reflection);
			if ($type && !class_exists($type) && !interface_exists($type)) {
				throw new ServiceCreationException(sprintf("Class or interface '%s' not found. Is return type of %s() correct?", $type, Nette\Utils\Callback::toString($entity)));
			}
			return $type;

		} elseif ($entity instanceof Reference) { // alias or factory
			return $this->resolveReferenceType($entity);

		} elseif (is_string($entity)) { // class
			if (!class_exists($entity)) {
				throw new ServiceCreationException("Class $entity not found.");
			}
			return $entity;
		}
		return null;
	}


	public function completeDefinition(Definition $def): void
	{
		$this->currentService = null;
		try {
			$def->complete($this);

		} catch (\Exception $e) {
			throw $this->completeException($e, $def);

		} finally {
			$this->currentService = null;
			if ($def->getType()) {
				$this->addDependency(new \ReflectionClass($def->getType()));
			}
		}
	}


	public function completeStatement(Statement $statement, Definition $currentService = null): Statement
	{
		$this->currentService = $currentService;

		$entity = $this->normalizeEntity($statement->getEntity());
		$arguments = $this->convertReferences($statement->arguments);

		switch (true) {
			case is_string($entity) && Strings::contains($entity, '?'): // PHP literal
			case $entity === 'types':
			case $entity === 'tags':
			case $entity === 'not':
				break;

			case is_string($entity): // create class
				if (!class_exists($entity)) {
					throw new ServiceCreationException("Class $entity not found.");
				} elseif ((new ReflectionClass($entity))->isAbstract()) {
					throw new ServiceCreationException("Class $entity is abstract.");
				} elseif (($rm = (new ReflectionClass($entity))->getConstructor()) !== null && !$rm->isPublic()) {
					$visibility = $rm->isProtected() ? 'protected' : 'private';
					throw new ServiceCreationException("Class $entity has $visibility constructor.");
				} elseif ($constructor = (new ReflectionClass($entity))->getConstructor()) {
					$arguments = $this->autowireArguments($constructor, $arguments);
				} elseif ($arguments) {
					throw new ServiceCreationException("Unable to pass arguments, class $entity has no constructor.");
				}
				break;

			case $entity instanceof Reference:
				$params = [];
				foreach ($this->getDefinition($entity->getName())->parameters as $k => $v) {
					$params[] = preg_replace('#\w+\z#', '\$$0', (is_int($k) ? $v : $k)) . (is_int($k) ? '' : ' = ' . PhpHelpers::dump($v));
				}
				$rm = new \ReflectionFunction(eval('return function(' . implode(', ', $params) . ') {};'));
				$arguments = $this->autowireArguments($rm, $arguments);
				break;

			case is_array($entity):
				if (!preg_match('#^\$?(\\\\?' . PhpHelpers::PHP_IDENT . ')+(\[\])?\z#', $entity[1])) {
					throw new ServiceCreationException("Expected function, method or property name, '$entity[1]' given.");
				}

				switch (true) {
					case $entity[0] === '': // function call
						if (!Nette\Utils\Arrays::isList($arguments)) {
							throw new ServiceCreationException("Unable to pass specified arguments to $entity[0].");
						} elseif (!function_exists($entity[1])) {
							throw new ServiceCreationException("Function $entity[1] doesn't exist.");
						}
						$rf = new \ReflectionFunction($entity[1]);
						$arguments = $this->autowireArguments($rf, $arguments);
						break;

					case $entity[0] instanceof Statement:
						$entity[0] = $this->completeStatement($entity[0], $currentService);
						// break omitted

					case is_string($entity[0]): // static method call
					case $entity[0] instanceof Reference:
						if ($entity[1][0] === '$') { // property getter, setter or appender
							Validators::assert($arguments, 'list:0..1', "setup arguments for '" . Nette\Utils\Callback::toString($entity) . "'");
							if (!$arguments && substr($entity[1], -2) === '[]') {
								throw new ServiceCreationException("Missing argument for $entity[1].");
							}
						} elseif (
							$type = !$entity[0] instanceof Reference || $entity[1] === 'create'
								? $this->resolveEntityType($entity[0])
								: $this->resolveServiceType($entity[0]->getName())
						) {
							$rc = new ReflectionClass($type);
							if ($rc->hasMethod($entity[1])) {
								$rm = $rc->getMethod($entity[1]);
								if (!$rm->isPublic()) {
									throw new ServiceCreationException("$type::$entity[1]() is not callable.");
								}
								$arguments = $this->autowireArguments($rm, $arguments);

							} elseif (!Nette\Utils\Arrays::isList($arguments)) {
								throw new ServiceCreationException("Unable to pass specified arguments to $type::$entity[1]().");
							}
						}
				}
		}

		$arguments = $this->completeArguments($arguments, $entity);

		if ($entity instanceof Reference) {
			$entity = $this->selfishReference($entity);
		} elseif (is_array($entity) && isset($entity[0]) && $entity[0] instanceof Reference) {
			$entity[0] = $this->selfishReference($entity[0]);
		}

		return new Statement($entity, $arguments);
	}


	private function completeArguments(array $arguments, $entity): array
	{
		try {
			array_walk_recursive($arguments, function (&$val): void {
				if ($val instanceof Statement) {
					$val = $this->completeStatement($val, $this->currentService);

				} elseif ($val instanceof Definition) {
					$val = $this->selfishReference($this->definitionToReference($val));

				} elseif ($val instanceof Reference) {
					$val = $this->selfishReference($this->normalizeReference($val));
				}
			});

		} catch (ServiceCreationException $e) {
			$toText = function ($x) { return $x instanceof Reference ? '@' . $x->getValue() : $x; } ;
			if ((is_string($entity) || $entity instanceof Reference || is_array($entity)) && !strpos($e->getMessage(), ' (used in')) {
				$desc = (is_string($entity) || $entity instanceof Reference)
					? $toText($entity) . '::__construct'
					: ((is_string($entity[0]) || $entity[0] instanceof Reference) ? ($toText($entity[0]) . '::') : 'method ') . $entity[1];
				$e->setMessage($e->getMessage() . " (used in $desc)");
			}
			throw $e;
		}
		return $arguments;
	}


	/**
	 * Creates a list of arguments using autowiring.
	 */
	private function autowireArguments(\ReflectionFunctionAbstract $function, array $arguments): array
	{
		if (!$function->isClosure()) {
			$this->addDependency($function);
		}
		return Autowiring::completeArguments($function, $arguments, $this);
	}


	/**
	 * @return string|array|Reference  literal, Class, Reference, [Class, member], [, globalFunc], [Reference, member], [Statement, member]
	 */
	private function normalizeEntity($entity)
	{
		if (is_array($entity)) {
			$item = &$entity[0];
		} else {
			$item = &$entity;
		}

		if ($item instanceof Definition) { // Definition -> Reference
			$item = $this->definitionToReference($item);

		} elseif ($ref = $this->normalizeReference($item)) { // Reference -> resolved Reference
			$item = $ref;
		}

		return $entity;
	}


	/**
	 * Converts @service or @\Class -> service name and checks its existence.
	 */
	public function normalizeReference($arg): ?Reference
	{
		if ($arg instanceof Reference) {
			$service = $arg->getValue();
		} elseif (is_string($arg) && preg_match('#^@[\w\\\\.][^:]*\z#', $arg)) {
			$service = substr($arg, 1);
		} else {
			return null;
		}
		if ($service === ContainerBuilder::THIS_SERVICE) {
			if (current(array_keys($this->builder->getDefinitions(), $this->currentService, true))) {
				return new Reference($this->currentService->getName());
			}
			return new Reference(ContainerBuilder::THIS_SERVICE);
		}
		if (Strings::contains($service, '\\')) {
			try {
				$res = $this->getByType($service);
			} catch (NotAllowedDuringResolvingException $e) {
				return new Reference($service);
			}
			if (!$res) {
				throw new ServiceCreationException("Reference to missing service of type $service.");
			}
			return new Reference($res);
		}
		if (!$this->builder->hasDefinition($service)) {
			throw new ServiceCreationException("Reference to missing service '$service'.");
		}
		return new Reference($service);
	}


	/**
	 * Resolves service name by type.
	 */
	public function getByType(string $type): ?string
	{
		if ($this->currentService && is_a($this->currentService->getType(), $type, true)) {
			return $this->currentService->getName();
		}
		return $this->builder->getByType($type, false);
	}


	private function definitionToReference(Definition $def): Reference
	{
		$name = current(array_keys($this->builder->getDefinitions(), $def, true));
		if (!$name) {
			throw new ServiceCreationException("Service '{$def->getName()}' is not added to ContainerBuilder.");
		}
		return new Reference($name);
	}


	/**
	 * Adds item to the list of dependencies.
	 * @param  \ReflectionClass|\ReflectionFunctionAbstract|string  $dep
	 * @return static
	 */
	public function addDependency($dep)
	{
		$this->builder->addDependency($dep);
		return $this;
	}


	private function selfishReference(Reference $ref): Reference
	{
		return $this->currentService && $ref->getValue() === $this->currentService->getName()
			? new Reference(ContainerBuilder::THIS_SERVICE)
			: $ref;
	}


	private function completeException(\Exception $e, Definition $def): ServiceCreationException
	{
		if ($e instanceof ServiceCreationException && Strings::startsWith($e->getMessage(), "Service '")) {
			return $e;
		} else {
			$message = "Service '{$def->getName()}'" . ($def->getType() ? " (type of {$def->getType()})" : '') . ': ' . $e->getMessage();
			return $e instanceof ServiceCreationException
				? $e->setMessage($message)
				: new ServiceCreationException($message, 0, $e);
		}
	}


	private function convertReferences(array $arguments): array
	{
		array_walk_recursive($arguments, function (&$val): void {
			if (is_string($val) && strlen($val) > 1 && $val[0] === '@' && $val[1] !== '@') {
				$pair = explode('::', substr($val, 1), 2);
				if (!isset($pair[1])) { // @service
					$val = new Reference($pair[0]);
				} elseif (preg_match('#^[A-Z][A-Z0-9_]*\z#', $pair[1], $m)) { // @service::CONSTANT
					$type = $this->resolveReferenceType(new Reference($pair[0]));
					$val = ContainerBuilder::literal($type . '::' . $pair[1]);
				} else { // @service::property
					$val = new Statement([new Reference($pair[0]), '$' . $pair[1]]);
				}

			} elseif (is_string($val) && substr($val, 0, 2) === '@@') { // escaped text @@
				$val = substr($val, 1);
			}
		});
		return $arguments;
	}
}
