<?php

/**
 * This file is part of the Nette Framework (https://nette.org)
 * Copyright (c) 2004 David Grudl (https://davidgrudl.com)
 */

declare(strict_types=1);

namespace Nette\DI\Definitions;

use Nette;
use Nette\DI\ServiceCreationException;
use Nette\Utils\Reflection;


/**
 * Definition of standard service.
 */
final class FactoryDefinition extends Definition
{
	private const METHOD_CREATE = 'create';

	/** @var array */
	public $parameters = [];

	/** @var Definition */
	private $returnedService;


	public function __construct()
	{
		$this->returnedService = new ServiceDefinition;
	}


	/**
	 * @return static
	 */
	public function setImplement(?string $type)
	{
		if ($type !== null && !interface_exists($type)) {
			throw new Nette\InvalidArgumentException("Service '{$this->getName()}': Interface '$type' not found.");
		}
		$rc = new \ReflectionClass($type);
		$method = $rc->hasMethod(self::METHOD_CREATE) ? $rc->getMethod(self::METHOD_CREATE) : null;
		if (count($rc->getMethods()) !== 1 || !$method || $method->isStatic()) {
			throw new Nette\InvalidArgumentException("Interface $type must have just one non-static method create() or get().");
		}
		return parent::setType($type);
	}


	public function getImplement(): ?string
	{
		return $this->getType();
	}


	/**
	 * @return static
	 */
	public function setReturnedType(?string $type)
	{
		$this->returnedService->setType($type);
		return $this;
	}


	/**
	 * @return ServiceDefinition
	 */
	final public function getReturnedType(): ?string
	{
		return $this->returnedService->getType();
	}


	/**
	 * @return static
	 */
	public function setReturnedService(Definition $definition)
	{
		$this->returnedService = $definition;
		return $this;
	}


	public function getReturnedService(): Definition
	{
		return $this->returnedService;
	}


	/**
	 * @deprecated Use getReturnedService()->setFactory()
	 */
	public function setFactory($factory, array $args = [])
	{
		$this->returnedService->setFactory($factory, $args);
		return $this;
	}


	/**
	 * @deprecated Use getReturnedService()->getFactory()
	 */
	public function getFactory(): ?Statement
	{
		return $this->returnedService->getFactory();
	}


	/**
	 * @deprecated Use getReturnedService()->getEntity()
	 */
	public function getEntity()
	{
		return $this->returnedService->getEntity();
	}


	/**
	 * @deprecated Use getReturnedService()->setArguments()
	 */
	public function setArguments(array $args = [])
	{
		$this->returnedService->setArguments($args);
		return $this;
	}


	/**
	 * @deprecated Use getReturnedService()->setSetup()
	 */
	public function setSetup(array $setup)
	{
		$this->returnedService->setSetup($setup);
		return $this;
	}


	/**
	 * @deprecated Use getReturnedService()->getSetup()
	 */
	public function getSetup(): array
	{
		return $this->returnedService->getSetup();
	}


	/**
	 * @deprecated Use getReturnedService()->addSetup()
	 */
	public function addSetup($entity, array $args = [])
	{
		$this->returnedService->addSetup($entity, $args);
		return $this;
	}


	/**
	 * @return static
	 */
	public function setParameters(array $params)
	{
		$this->parameters = $params;
		return $this;
	}


	public function getParameters(): array
	{
		return $this->parameters;
	}


	public function resolveType(Nette\DI\Resolver $resolver): void
	{
		$interface = $this->getType();
		$rc = new \ReflectionClass($interface);
		$method = $rc->getMethod(self::METHOD_CREATE);

		$service = $this->returnedService;

		if (!$service->getType() && !$service->getEntity()) {
			$returnType = Nette\DI\Helpers::getReturnType($method);
			if (!$returnType) {
				throw new ServiceCreationException("Method $interface::create() has not return type hint or annotation @return.");
			} elseif (!class_exists($returnType)) {
				throw new ServiceCreationException("Check a type hint or annotation @return of the $interface::create() method, class '$returnType' cannot be found.");
			}
			$service->setType($returnType);
		}

		if (!$service->getEntity()) {
			$service->setFactory($service->getType(), $service->getFactory() ? $service->getFactory()->arguments : []);
		}

		$this->returnedService->resolveType($resolver);
	}


	public function complete(Nette\DI\Resolver $resolver): void
	{
		$service = $this->returnedService;

		if (!$this->parameters) {
			$interface = $this->getType();
			$method = new \ReflectionMethod($interface, self::METHOD_CREATE);

			$ctorParams = [];
			if (
				($class = $resolver->resolveEntityType($service->getFactory()))
				&& ($ctor = (new \ReflectionClass($class))->getConstructor())
			) {
				foreach ($ctor->getParameters() as $param) {
					$ctorParams[$param->getName()] = $param;
				}
			}

			foreach ($method->getParameters() as $param) {
				$hint = Reflection::getParameterType($param);
				if (isset($ctorParams[$param->getName()])) {
					$arg = $ctorParams[$param->getName()];
					$argHint = Reflection::getParameterType($arg);
					if ($hint !== $argHint && !is_a($hint, $argHint, true)) {
						throw new ServiceCreationException("Type hint for \${$param->getName()} in $interface::create() doesn't match type hint in $class constructor.");
					}
					$service->getFactory()->arguments[$arg->getPosition()] = Nette\DI\ContainerBuilder::literal('$' . $arg->getName());

				} elseif (!$service->getSetup()) {
					$hint = Nette\Utils\ObjectHelpers::getSuggestion(array_keys($ctorParams), $param->getName());
					throw new ServiceCreationException("Unused parameter \${$param->getName()} when implementing method $interface::create()" . ($hint ? ", did you mean \${$hint}?" : '.'));
				}
				$nullable = $hint && $param->allowsNull() && (!$param->isDefaultValueAvailable() || $param->getDefaultValue() !== null);
				$paramDef = ($nullable ? '?' : '') . $hint . ' ' . $param->getName();
				if ($param->isDefaultValueAvailable()) {
					$this->parameters[$paramDef] = Reflection::getParameterDefaultValue($param);
				} else {
					$this->parameters[] = $paramDef;
				}
			}
		}

		$service
			->setName($this->getName())
			->setImplementMode('create')
			->complete($resolver);
	}


	public function generateMethod(Nette\PhpGenerator\Method $method, Nette\DI\PhpGenerator $generator): void
	{
		$class = (new Nette\PhpGenerator\ClassType)
			->addImplement($this->getType());

		$class->addProperty('container')
			->setVisibility('private');

		$class->addMethod('__construct')
			->addBody('$this->container = $container;')
			->addParameter('container')
			->setTypeHint($generator->getClassName());

		$rm = new \ReflectionMethod($this->getType(), self::METHOD_CREATE);

		$methodCreate = $class->addMethod(self::METHOD_CREATE);
		$this->returnedService->generateMethod($methodCreate, $generator);
		$methodCreate
			->setParameters($generator->convertParameters($this->parameters))
			->setReturnType(Reflection::getReturnType($rm))
			->setBody(str_replace('$this', '$this->container', $methodCreate->getBody()));

		$method->setBody('return new class ($this) ' . $class . ';');
	}


	public function __clone()
	{
		parent::__clone();
		$this->returnedService = unserialize(serialize($this->returnedService));
	}
}
