<?php

/**
 * @see       https://github.com/laminas/laminas-validator for the canonical source repository
 * @copyright https://github.com/laminas/laminas-validator/blob/master/COPYRIGHT.md
 * @license   https://github.com/laminas/laminas-validator/blob/master/LICENSE.md New BSD License
 */

namespace LaminasTest\Validator;

use Laminas\Validator\Hex;
use PHPUnit\Framework\TestCase;

/**
 * @group      Laminas_Validator
 */
class HexTest extends TestCase
{
    /**
     * @var Hex
     */
    protected $validator;

    protected function setUp() : void
    {
        $this->validator = new Hex();
    }

    /**
     * Ensures that the validator follows expected behavior
     *
     * @dataProvider basicDataProvider
     * @param int|string $input
     * @return void
     */
    public function testBasic($input, bool $expected)
    {
        $this->assertSame($expected, $this->validator->isValid($input));
    }

    public function basicDataProvider()
    {
        return [
            // phpcs:disable
            'valid; int; 1' => [1, true],
            'valid; hex; 0x1' => [0x1, true],
            'valid; hex; 0x123' => [0x123, true],
            'valid; string; 1' => ['1', true],
            'valid; string; abc123' => ['abc123', true],
            'valid; string; ABC123' => ['ABC123', true],
            'valid; string; 1234567890abcdef' => ['1234567890abcdef', true],

            'invalid; string; g' => ['g', false],
            'invalid; string; 1.2' => ['1.2', false],
            // phpcs:enable
        ];
    }

    /**
     * Ensures that getMessages() returns expected default value
     *
     * @return void
     */
    public function testGetMessages()
    {
        $this->assertEquals([], $this->validator->getMessages());
    }

    /**
     * @Laminas-4352
     */
    public function testNonStringValidation()
    {
        $this->assertFalse($this->validator->isValid([1 => 1]));
    }

    public function testEqualsMessageTemplates()
    {
        $validator = $this->validator;
        $this->assertAttributeEquals(
            $validator->getOption('messageTemplates'),
            'messageTemplates',
            $validator
        );
    }
}
