<?php

declare(strict_types=1);

namespace LaminasTest\Validator\File;

use Laminas\Validator\Exception\InvalidArgumentException;
use Laminas\Validator\File;
use PHPUnit\Framework\TestCase;

use function array_merge;
use function basename;
use function current;
use function is_array;

use const UPLOAD_ERR_NO_FILE;

/**
 * @group      Laminas_Validator
 */
class ImageSizeTest extends TestCase
{
    /**
     * @psalm-return array<array-key, array{
     *     0: array<string, int>,
     *     1: string|array{
     *         tmp_name: string,
     *         name: string,
     *         size: int,
     *         error: int,
     *         type: string
     *     },
     *     2: bool,
     *     3: string|string[]
     * }>
     */
    public function basicBehaviorDataProvider(): array
    {
        $testFile     = __DIR__ . '/_files/picture.jpg';
        $pictureTests = [
            //    Options, isValid Param, Expected value, Expected message
            [
                ['minWidth' => 0,   'minHeight' => 10,  'maxWidth' => 1000, 'maxHeight' => 2000],
                $testFile,
                true,
                '',
            ],
            [
                ['minWidth' => 0,   'minHeight' => 0,   'maxWidth' => 200,  'maxHeight' => 200],
                $testFile,
                true,
                '',
            ],
            [
                ['minWidth' => 150, 'minHeight' => 150, 'maxWidth' => 200,  'maxHeight' => 200],
                $testFile,
                false,
                ['fileImageSizeWidthTooSmall', 'fileImageSizeHeightTooSmall'],
            ],
            [
                ['minWidth' => 80,  'minHeight' => 0,   'maxWidth' => 80,   'maxHeight' => 200],
                $testFile,
                true,
                '',
            ],
            [
                ['minWidth' => 0,   'minHeight' => 0,   'maxWidth' => 60,   'maxHeight' => 200],
                $testFile,
                false,
                'fileImageSizeWidthTooBig',
            ],
            [
                ['minWidth' => 90,  'minHeight' => 0,   'maxWidth' => 200,  'maxHeight' => 200],
                $testFile,
                false,
                'fileImageSizeWidthTooSmall',
            ],
            [
                ['minWidth' => 0,   'minHeight' => 0,   'maxWidth' => 200,  'maxHeight' => 80],
                $testFile,
                false,
                'fileImageSizeHeightTooBig',
            ],
            [
                ['minWidth' => 0,   'minHeight' => 110, 'maxWidth' => 200,  'maxHeight' => 140],
                $testFile,
                false,
                'fileImageSizeHeightTooSmall',
            ],
        ];

        $testFile    = __DIR__ . '/_files/nofile.mo';
        $noFileTests = [
            //    Options, isValid Param, Expected value, message
            [
                ['minWidth' => 0, 'minHeight' => 10, 'maxWidth' => 1000, 'maxHeight' => 2000],
                $testFile,
                false,
                'fileImageSizeNotReadable',
            ],
        ];

        $testFile    = __DIR__ . '/_files/badpicture.jpg';
        $badPicTests = [
            //    Options, isValid Param, Expected value, message
            [
                ['minWidth' => 0, 'minHeight' => 10, 'maxWidth' => 1000, 'maxHeight' => 2000],
                $testFile,
                false,
                'fileImageSizeNotDetected',
            ],
        ];

        // Dupe data in File Upload format
        $testData = array_merge($pictureTests, $noFileTests, $badPicTests);
        foreach ($testData as $data) {
            $fileUpload = [
                'tmp_name' => $data[1],
                'name'     => basename($data[1]),
                'size'     => 200,
                'error'    => 0,
                'type'     => 'text',
            ];
            $testData[] = [$data[0], $fileUpload, $data[2], $data[3]];
        }
        return $testData;
    }

    /**
     * Ensures that the validator follows expected behavior
     *
     * @dataProvider basicBehaviorDataProvider
     * @param string|array $isValidParam
     * @param string|string[] $messageKeys
     */
    public function testBasic(array $options, $isValidParam, bool $expected, $messageKeys): void
    {
        $validator = new File\ImageSize($options);
        $this->assertEquals($expected, $validator->isValid($isValidParam));
        if (! $expected) {
            if (! is_array($messageKeys)) {
                $messageKeys = [$messageKeys];
            }
            foreach ($messageKeys as $messageKey) {
                $this->assertArrayHasKey($messageKey, $validator->getMessages());
            }
        }
    }

    /**
     * Ensures that the validator follows expected behavior for legacy Laminas\Transfer API
     *
     * @dataProvider basicBehaviorDataProvider
     * @param string|array $isValidParam
     * @param string|string[] $messageKeys
     */
    public function testLegacy(array $options, $isValidParam, bool $expected, $messageKeys): void
    {
        if (! is_array($isValidParam)) {
            $this->markTestSkipped('An array is expected for legacy compat tests');
        }

        $validator = new File\ImageSize($options);
        $this->assertEquals($expected, $validator->isValid($isValidParam['tmp_name'], $isValidParam));
        if (! $expected) {
            if (! is_array($messageKeys)) {
                $messageKeys = [$messageKeys];
            }
            foreach ($messageKeys as $messageKey) {
                $this->assertArrayHasKey($messageKey, $validator->getMessages());
            }
        }
    }

    /**
     * Ensures that getImageMin() returns expected value
     */
    public function testGetImageMin(): void
    {
        $validator = new File\ImageSize(['minWidth' => 1, 'minHeight' => 10, 'maxWidth' => 100, 'maxHeight' => 1000]);
        $this->assertEquals(['minWidth' => 1, 'minHeight' => 10], $validator->getImageMin());

        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('greater than or equal');
        new File\ImageSize(['minWidth' => 1000, 'minHeight' => 100, 'maxWidth' => 10, 'maxHeight' => 1]);
    }

    /**
     * Ensures that setImageMin() returns expected value
     */
    public function testSetImageMin(): void
    {
        $validator = new File\ImageSize([
            'minWidth'  => 100,
            'minHeight' => 1000,
            'maxWidth'  => 10000,
            'maxHeight' => 100000,
        ]);
        $validator->setImageMin(['minWidth' => 10, 'minHeight' => 10]);
        $this->assertEquals(['minWidth' => 10, 'minHeight' => 10], $validator->getImageMin());

        $validator->setImageMin(['minWidth' => 9, 'minHeight' => 100]);
        $this->assertEquals(['minWidth' => 9, 'minHeight' => 100], $validator->getImageMin());

        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('less than or equal');
        $validator->setImageMin(['minWidth' => 20000, 'minHeight' => 20000]);
    }

    /**
     * Ensures that getImageMax() returns expected value
     */
    public function testGetImageMax(): void
    {
        $validator = new File\ImageSize([
            'minWidth'  => 10,
            'minHeight' => 100,
            'maxWidth'  => 1000,
            'maxHeight' => 10000,
        ]);
        $this->assertEquals(['maxWidth' => 1000, 'maxHeight' => 10000], $validator->getImageMax());

        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('greater than or equal');
        new File\ImageSize([
            'minWidth'  => 10000,
            'minHeight' => 1000,
            'maxWidth'  => 100,
            'maxHeight' => 10,
        ]);
    }

    /**
     * Ensures that setImageMax() returns expected value
     */
    public function testSetImageMax(): void
    {
        $validator = new File\ImageSize([
            'minWidth'  => 10,
            'minHeight' => 100,
            'maxWidth'  => 1000,
            'maxHeight' => 10000,
        ]);
        $validator->setImageMax(['maxWidth' => 100, 'maxHeight' => 100]);
        $this->assertEquals(['maxWidth' => 100, 'maxHeight' => 100], $validator->getImageMax());

        $validator->setImageMax(['maxWidth' => 110, 'maxHeight' => 1000]);
        $this->assertEquals(['maxWidth' => 110, 'maxHeight' => 1000], $validator->getImageMax());

        $validator->setImageMax(['maxHeight' => 1100]);
        $this->assertEquals(['maxWidth' => 110, 'maxHeight' => 1100], $validator->getImageMax());

        $validator->setImageMax(['maxWidth' => 120]);
        $this->assertEquals(['maxWidth' => 120, 'maxHeight' => 1100], $validator->getImageMax());

        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('greater than or equal');
        $validator->setImageMax(['maxWidth' => 10000, 'maxHeight' => 1]);
    }

    /**
     * Ensures that getImageWidth() returns expected value
     */
    public function testGetImageWidth(): void
    {
        $validator = new File\ImageSize(['minWidth' => 1, 'minHeight' => 10, 'maxWidth' => 100, 'maxHeight' => 1000]);
        $this->assertEquals(['minWidth' => 1, 'maxWidth' => 100], $validator->getImageWidth());
    }

    /**
     * Ensures that setImageWidth() returns expected value
     */
    public function testSetImageWidth(): void
    {
        $validator = new File\ImageSize([
            'minWidth'  => 100,
            'minHeight' => 1000,
            'maxWidth'  => 10000,
            'maxHeight' => 100000,
        ]);
        $validator->setImageWidth(['minWidth' => 2000, 'maxWidth' => 2200]);
        $this->assertEquals(['minWidth' => 2000, 'maxWidth' => 2200], $validator->getImageWidth());

        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('less than or equal');
        $validator->setImageWidth(['minWidth' => 20000, 'maxWidth' => 200]);
    }

    /**
     * Ensures that getImageHeight() returns expected value
     */
    public function testGetImageHeight(): void
    {
        $validator = new File\ImageSize(['minWidth' => 1, 'minHeight' => 10, 'maxWidth' => 100, 'maxHeight' => 1000]);
        $this->assertEquals(['minHeight' => 10, 'maxHeight' => 1000], $validator->getImageHeight());
    }

    /**
     * Ensures that setImageHeight() returns expected value
     */
    public function testSetImageHeight(): void
    {
        $validator = new File\ImageSize([
            'minWidth'  => 100,
            'minHeight' => 1000,
            'maxWidth'  => 10000,
            'maxHeight' => 100000,
        ]);
        $validator->setImageHeight(['minHeight' => 2000, 'maxHeight' => 2200]);
        $this->assertEquals(['minHeight' => 2000, 'maxHeight' => 2200], $validator->getImageHeight());

        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('less than or equal');
        $validator->setImageHeight(['minHeight' => 20000, 'maxHeight' => 200]);
    }

    /**
     * @group Laminas-11258
     */
    public function testLaminas11258(): void
    {
        $validator = new File\ImageSize([
            'minWidth'  => 100,
            'minHeight' => 1000,
            'maxWidth'  => 10000,
            'maxHeight' => 100000,
        ]);
        $this->assertFalse($validator->isValid(__DIR__ . '/_files/nofile.mo'));
        $this->assertArrayHasKey('fileImageSizeNotReadable', $validator->getMessages());
        $this->assertStringContainsString('does not exist', current($validator->getMessages()));
    }

    public function testEmptyFileShouldReturnFalseAndDisplayNotFoundMessage(): void
    {
        $validator = new File\ImageSize();

        $this->assertFalse($validator->isValid(''));
        $this->assertArrayHasKey(File\ImageSize::NOT_READABLE, $validator->getMessages());

        $filesArray = [
            'name'     => '',
            'size'     => 0,
            'tmp_name' => '',
            'error'    => UPLOAD_ERR_NO_FILE,
            'type'     => '',
        ];

        $this->assertFalse($validator->isValid($filesArray));
        $this->assertArrayHasKey(File\ImageSize::NOT_READABLE, $validator->getMessages());
    }

    public function testConstructorCanAcceptAllOptionsAsDiscreteArguments(): void
    {
        $minWidth  = 20;
        $minHeight = 10;
        $maxWidth  = 200;
        $maxHeight = 100;
        $validator = new File\ImageSize($minWidth, $minHeight, $maxWidth, $maxHeight);

        $this->assertSame($minWidth, $validator->getMinWidth());
        $this->assertSame($minHeight, $validator->getMinHeight());
        $this->assertSame($maxWidth, $validator->getMaxWidth());
        $this->assertSame($maxHeight, $validator->getMaxHeight());
    }

    public function testIsValidRaisesExceptionForArrayNotInFilesFormat(): void
    {
        $validator = new File\ImageSize([
            'minWidth'  => 100,
            'minHeight' => 1000,
            'maxWidth'  => 10000,
            'maxHeight' => 100000,
        ]);
        $value     = ['foo' => 'bar'];

        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('Value array must be in $_FILES format');

        $validator->isValid($value);
    }
}
