<?php

declare(strict_types=1);

namespace LaminasTest\I18n\Filter;

use Laminas\I18n\Filter\NumberFormat as NumberFormatFilter;
use LaminasTest\I18n\TestCase;
use NumberFormatter;
use stdClass;

class NumberFormatTest extends TestCase
{
    public function testConstructWithOptions(): void
    {
        $filter = new NumberFormatFilter([
            'locale' => 'en_US',
            'style'  => NumberFormatter::DECIMAL,
        ]);

        self::assertEquals('en_US', $filter->getLocale());
        self::assertEquals(NumberFormatter::DECIMAL, $filter->getStyle());
    }

    public function testConstructWithParameters(): void
    {
        $filter = new NumberFormatFilter('en_US', NumberFormatter::DECIMAL);

        self::assertEquals('en_US', $filter->getLocale());
        self::assertEquals(NumberFormatter::DECIMAL, $filter->getStyle());
    }

    /** @return array<array-key, array{0: string, 1: int, 2: int, 3: float, 4: string}> */
    public function numberToFormattedProvider(): array
    {
        return [
            [
                'en_US',
                NumberFormatter::DEFAULT_STYLE,
                NumberFormatter::TYPE_DOUBLE,
                1234567.8912346,
                '1,234,567.891',
            ],
            [
                'de_DE',
                NumberFormatter::DEFAULT_STYLE,
                NumberFormatter::TYPE_DOUBLE,
                1234567.8912346,
                '1.234.567,891',
            ],
            [
                'ru_RU',
                NumberFormatter::DEFAULT_STYLE,
                NumberFormatter::TYPE_DOUBLE,
                1234567.8912346,
                '1 234 567,891',
            ],
        ];
    }

    /**
     * @dataProvider numberToFormattedProvider
     */
    public function testNumberToFormatted(string $locale, int $style, int $type, float $value, string $expected): void
    {
        $filter = new NumberFormatFilter($locale, $style, $type);
        self::assertEquals($expected, $filter->filter($value));
    }

    /** @return array<array-key, array{0: string, 1: int, 2: int, 3: string, 4: float}> */
    public function formattedToNumberProvider(): array
    {
        return [
            [
                'en_US',
                NumberFormatter::DEFAULT_STYLE,
                NumberFormatter::TYPE_DOUBLE,
                '1,234,567.891',
                1234567.891,
            ],
            [
                'de_DE',
                NumberFormatter::DEFAULT_STYLE,
                NumberFormatter::TYPE_DOUBLE,
                '1.234.567,891',
                1234567.891,
            ],
            [
                'ru_RU',
                NumberFormatter::DEFAULT_STYLE,
                NumberFormatter::TYPE_DOUBLE,
                '1 234 567,891',
                1234567.891,
            ],
        ];
    }

    /**
     * @dataProvider formattedToNumberProvider
     */
    public function testFormattedToNumber(string $locale, int $style, int $type, string $value, float $expected): void
    {
        $filter = new NumberFormatFilter($locale, $style, $type);
        self::assertEquals($expected, $filter->filter($value));
    }

    /** @return array<array-key, array{0: mixed}> */
    public function returnUnfilteredDataProvider(): array
    {
        return [
            [null],
            [new stdClass()],
            [
                [
                    '1.234.567,891',
                    '1.567,891',
                ],
            ],
        ];
    }

    /**
     * @dataProvider returnUnfilteredDataProvider
     * @param mixed $input
     */
    public function testReturnUnfiltered($input): void
    {
        $filter = new NumberFormatFilter('de_AT', NumberFormatter::DEFAULT_STYLE, NumberFormatter::TYPE_DOUBLE);

        self::assertEquals($input, $filter->filter($input));
    }
}
