<?php

declare(strict_types=1);

namespace AsyncAws\Core\Credentials;

use AsyncAws\Core\Configuration;
use Psr\Cache\CacheItemPoolInterface;

/**
 * Cache the Credential generated by the decorated CredentialProvider with PSR-6.
 *
 * The Credential will be reused until it expires.
 *
 * @author Jérémy Derussé <jeremy@derusse.com>
 */
final class PsrCacheProvider implements CredentialProvider
{
    private $cache;

    private $decorated;

    public function __construct(CredentialProvider $decorated, CacheItemPoolInterface $cache)
    {
        $this->decorated = $decorated;
        $this->cache = $cache;
    }

    public function getCredentials(Configuration $configuration): ?Credentials
    {
        $item = $this->cache->getItem('AsyncAws.Credentials.' . sha1(\serialize([$configuration, \get_class($this->decorated)])));
        if (!$item->isHit()) {
            $item->set($credential = $this->decorated->getCredentials($configuration));

            if (null !== $credential && null !== $exp = $credential->getExpireDate()) {
                $item->expiresAt($exp);
                $this->cache->save($item);
            }
        }

        return $item->get();
    }
}
